/**
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS
 * FOR A PARTICULAR PURPOSE. THIS CODE AND INFORMATION ARE NOT SUPPORTED BY XEBIALABS.
 */
package com.xebialabs.deployit.plugin.codepipeline.helpers;

import com.amazonaws.auth.BasicSessionCredentials;
import com.amazonaws.services.codepipeline.model.*;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.model.S3Object;
import com.xebialabs.deployit.plugin.codepipeline.exceptions.CodePipelinePluginExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.UUID;

public abstract class S3Helper {

    private static final Logger logger = LoggerFactory.getLogger(S3Helper.class);

    public static Path downloadPackageFromJob(Job job) throws CodePipelinePluginExecutionException {

        Artifact artifact = job.getData().getInputArtifacts().get(0);

        logger.debug(String.format("Working on artifact with name %s and revision %s", artifact.getName(), artifact.getRevision()));
        S3Object sessionObject = getS3Object(artifact, job.getData().getArtifactCredentials());

        logger.debug("S3 object with build artifact " + sessionObject.toString());

        return downloadFile(sessionObject);
    }

    private static AmazonS3Client getS3Client(com.amazonaws.auth.AWSSessionCredentials sessionCredentials) {
        AmazonS3Client client = null;

        if (sessionCredentials != null) {
            client = new AmazonS3Client(sessionCredentials);
        }
        return client;
    }

    private static Path downloadFile(S3Object sessionObject) throws CodePipelinePluginExecutionException {
        try (InputStream inputStream = sessionObject.getObjectContent()) {
            Path filePath = Files.createTempFile(UUID.randomUUID().toString(), ".dar");
            Files.copy(inputStream, filePath, StandardCopyOption.REPLACE_EXISTING);
            logger.debug("downloaded S3 file to {}", filePath);
            return filePath;
        } catch (IOException e) {
            logger.error("Exception while downloading file from S3", e);
            throw new CodePipelinePluginExecutionException("Unable to download file from S3 bucket");
        }
    }

    private static S3Object getS3Object(Artifact artifact, AWSSessionCredentials awsSessionCredentials) {
        S3ArtifactLocation artifactLocation = artifact.getLocation().getS3Location();

        BasicSessionCredentials basicCredentials = new BasicSessionCredentials(
                awsSessionCredentials.getAccessKeyId(),
                awsSessionCredentials.getSecretAccessKey(),
                awsSessionCredentials.getSessionToken());

        AmazonS3Client client = getS3Client(basicCredentials);

        if (client == null) {
            throw new CodePipelinePluginExecutionException("No S3 Object Found!!");
        }

        String bucketName = artifactLocation.getBucketName();
        S3Object sessionObject = client.getObject(bucketName, artifactLocation.getObjectKey());

        return sessionObject;
    }
}
