#
# Copyright (c) 2020. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

import json
from overtherepy import OverthereHostSession
import sys
import re


class HelmRunner:

    def __init__(self, helmclient, cluster):
        self.helmclient = helmclient
        self._preview = False

    def get_helm_command(self, install_or_upgrade=False):
        helm = '{0}/helm'.format(self.helmclient.home)

        if self.helmclient.kubeContext is not None:
            helm = helm + \
                   ' --kube-context {0}'.format(self.helmclient.kubeContext)
        if self.helmclient.kubeConfig is not None:
            helm = helm + \
                   ' --kubeconfig {0}'.format(self.helmclient.kubeConfig)
        if self.helmclient.helmHost is not None:
            helm = helm + ' --host {0}'.format(self.helmclient.helmHost)

        if self.helmclient.debug:
            helm = helm + ' --debug'

        if install_or_upgrade:
            if self.helmclient.caFile is not None:
                helm = helm + ' --ca-file {0}'.format(self.helmclient.caFile)
            if self.helmclient.username is not None:
                helm = helm + ' --username {0}'.format(self.helmclient.username)
            if self.helmclient.password is not None:
                    helm = helm + ' --password ********'
        return helm

    def command_line(self, session, deployed):
        raise Exception("Not Implemented")

    # Masking credentials and secretInputVariables
    def mask_command_line(self, data, deployed=None):
        helm_password = "--password {0}".format(self.helmclient.password)
        if helm_password in data:
            helm_password_arr = helm_password.split(" ")
            helm_password_arr[1] = "*******"
            new_helm_password = " ".join(helm_password_arr)
            data = data.replace(helm_password, new_helm_password)
        if deployed:
            secret_variables = ["{0}={1}".format(k,v) for k, v in deployed.secretInputVariables.items()]
            for secret_variable in secret_variables:
                if secret_variable in data:
                    secret_variable_arr = secret_variable.split("=")
                    secret_variable_arr[1] = "*******"
                    new_secret_variable = " ".join(secret_variable_arr)
                    data = data.replace(secret_variable, new_secret_variable)

        return data

    # Masking response
    def mask_response(self, data, deployed=None):
        if deployed:
            lines = data.split("\n")
            for i, line in enumerate(lines):
                if line.lower().__contains__('password:'):
                   if(line.strip().__contains__(": ")):
                        password_varriable_arr = line.strip().split(": ")
                        password_varriable_arr[1] = "*******"
                        new_password_varriable_arr = ": ".join(password_varriable_arr)
                        data = data.replace(line, line.replace(line.strip(), new_password_varriable_arr))

        return data

    def preview(self, deployed):
        session = OverthereHostSession(self.helmclient.host, stream_command_output=False)
        try:
            self._preview = True
            command_line = self.command_line(session, deployed)
            masked_command_line = self.mask_command_line(command_line, deployed)
            print(masked_command_line)
        finally:
            session.close_conn()

    def execute(self, deployed):
        session = OverthereHostSession(self.helmclient.host, stream_command_output=False)
        try:
            command_line = self.command_line(session, deployed)
            masked_command_line = self.mask_command_line(command_line, deployed)
            print(masked_command_line)
            uploaded_runner = session.upload_text_content_to_work_dir(
                command_line, 'xldeploy_helm.sh', executable=True)
            print(uploaded_runner.path)
            response = session.execute(command_line, check_success=False)
            print(self.mask_response(str("\n".join(response.stdout)), deployed))
            print(self.mask_response(str("\n".join(response.stderr)), deployed))
            rc = response.rc
            if response.rc > 0:
                sys.exit(rc)
        finally:
            session.close_conn()

    def generate_variable(self, deployed):
        vars = ["--set {0}={1}".format(k, v) for k, v in deployed.inputVariables.items()]
        vars.extend(["--set {0}=********".format(k)
                         for k, v in deployed.secretInputVariables.items()])
        return " ".join(vars)


    def namespace(self, deployed):
        return deployed.container.namespaceName if deployed.container.hasProperty("namespaceName") else deployed.container.projectName


class HelmInstall(HelmRunner):

    def command_line(self, session, deployed):
        return "{0} upgrade --install {1} {2}".format(self.get_helm_command(install_or_upgrade=True), self.parameters(session, deployed),
                                                      "--insecure-skip-tls-verify" if (self.helmclient.insecureConnection and int(self.helmclient.version) == 3) else "")

    def parameters(self, session, deployed):
        values = {'chartName': deployed.chartName,
                  'namespace': self.namespace(deployed),
                  'name': deployed.name,
                  'chartVersion': deployed.chartVersion}

        if int(self.helmclient.version) == 2 or int(self.helmclient.version) == 3:
            parameters = "{name} {chartName} --namespace {namespace} --version {chartVersion}".format(**values)
        else:
            raise Exception("Unknown helm version {0}".format(self.helmclient.version))

        parameters = parameters + " " + self.generate_variable(deployed)

        for cf in deployed.configurationFiles:
            uploaded_file = session.upload_file_to_work_dir(cf.getFile())
            parameters = parameters + " -f " + uploaded_file.getPath()

        return parameters


class HelmUpgrade(HelmRunner):

    def command_line(self, session, deployed):
        return "{0} upgrade {1} {2}".format(self.get_helm_command(install_or_upgrade=True), self.parameters(session, deployed),
                                            "--insecure-skip-tls-verify" if (self.helmclient.insecureConnection and int(self.helmclient.version) == 3) else "")

    def parameters(self, session, deployed):
        values = {'chartName': deployed.chartName,
                  'namespace': self.namespace(deployed),
                  'name': deployed.name,
                  'chartVersion': deployed.chartVersion}

        if int(self.helmclient.version) == 2 or int(self.helmclient.version) == 3:
            parameters = "{name} {chartName} --namespace {namespace} --version {chartVersion}".format(**values)
        else:
            raise Exception("Unknown helm version {0}".format(self.helmclient.version))

        parameters = parameters + " " + self.generate_variable(deployed)

        for cf in deployed.configurationFiles:
            uploaded_file = session.upload_file_to_work_dir(cf.getFile())
            parameters = parameters + " -f " + uploaded_file.getPath()

        return parameters


class HelmAddRepo(HelmRunner):

    def command_line(self, session, deployed):
        return "{0} {1} {2} {3}".format(self.get_helm_command(install_or_upgrade=True), self.parameters(session, deployed),
                                    "--force-update" if self.helmclient.forceUpdate else "",
                                        "--insecure-skip-tls-verify" if (self.helmclient.insecureConnection and int(self.helmclient.version) == 3) else "")

    def parameters(self, session, deployed):
        values = {'chartName': deployed.chartName.split('/')[0],
                  'repositoryUrl': deployed.repositoryUrl}

        if int(self.helmclient.version) == 2 or int(self.helmclient.version) == 3:
            parameters = "repo add {chartName} {repositoryUrl}".format(**values)
        else:
            raise Exception("Unknown helm version {0}".format(self.helmclient.version))

        return parameters

class HelmTest(HelmRunner):

    def command_line(self, session, deployed):
        if deployed.container.hasProperty('namespaceName'):
            namespaceName = deployed.container.namespaceName
        else:
            namespaceName = deployed.container.projectName
        return "{0} test --namespace {1} {2}".format(self.get_helm_command(), namespaceName, deployed.name)

