#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import de.schlichtherle.truezip.file.TArchiveDetector as TArchiveDetector
import de.schlichtherle.truezip.file.TFile as TFile
import os.path
import string
import tempfile

from kubernetes import client


class ConfigMapHelper(object):
    @staticmethod
    def read_config_map(deployed):
        v1_config_map = client.V1ConfigMap()
        v1_config_map.metadata = client.V1ObjectMeta(name=ConfigMapHelper.get_config_map_name(deployed))
        v1_config_map.data = deployed.data.copy()
        data = ConfigMapHelper.get_config_map_files(deployed.propertyFiles)
        for folder in deployed.propertyFolders:
            dir_path = tempfile.mkdtemp()
            TFile.cp_rp(TFile(folder.file.path), TFile(dir_path), TArchiveDetector.NULL, TArchiveDetector.NULL)
            data = ConfigMapHelper.get_config_map_folder(dir_path, data)
        v1_config_map.data.update(data)
        return v1_config_map

    @staticmethod
    def get_config_map_name(deployed):
        return deployed.configMapName if deployed.configMapName else deployed.name

    @staticmethod
    def get_config_map_files(deployed_files):
        return dict((deployed_file.file.name, open(deployed_file.file.path).read()) for deployed_file in deployed_files)

    @staticmethod
    def get_referenced_resources(config_maps, deltas, type):
        noop_delta_pod_or_deployment = filter(
            lambda delta: delta.operation == "NOOP" and delta.deployedOrPrevious.type == type,
            deltas.deltas)
        candidates = []
        for delta_config_map in config_maps:
            for delta in noop_delta_pod_or_deployment:
                abstract_pod = delta.deployed
                if abstract_pod.restartOnConfigMapChanges and (ConfigMapHelper.__is_volume_candidate(delta_config_map,
                                                                                                     abstract_pod) or ConfigMapHelper.__is_environment_candidate(
                    delta_config_map, abstract_pod)):
                    candidates.append(delta)
        return set(candidates)

    @staticmethod
    def __is_volume_candidate(config_map_delta, abstract_pod_or_deployment):
        if abstract_pod_or_deployment.volumes:
            for volume in abstract_pod_or_deployment.volumes:
                if volume.volumeType == 'configMap' and 'name' in volume.properties and volume.properties[
                    'name'] == ConfigMapHelper.get_config_map_name(config_map_delta.deployed):
                    return True

    @staticmethod
    def __is_environment_candidate(config_map_delta, abstract_pod_or_deployment):
        if abstract_pod_or_deployment.containers:
            for container in abstract_pod_or_deployment.containers:
                if container.envVars:
                    for envVar in container.envVars:
                        if envVar.type == "k8s.envVar.ConfigMap" and (
                                envVar.configMapName == ConfigMapHelper.get_config_map_name(config_map_delta.deployed)):
                            return True
    
    @staticmethod
    def get_config_map_folder(root_dir, folder_dict):
        for lists in os.listdir(root_dir):
            path = os.path.join(root_dir, lists)
            if not os.path.isdir(path):
                folder_dict[string.replace(path, root_dir + os.sep, "")] = open(path).read()

        return folder_dict
