#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from kubernetes.client.rest import ApiException
from xld.kubernetes import KubernetesBaseClient


class KubernetesAppsClient(KubernetesBaseClient):
    def __init__(self, container):
        super(KubernetesAppsClient, self).__init__(container)

    def get_apps_api(self):
        print ("[Using API version: {}]".format(self.get_api_version()))
        return client.AppsV1beta2Api(self.internal_api_client)

    def get_api_version(self):
        return "apps/v1beta2"

    def read_deployment(self, name, namespace):
        return self.get_apps_api().read_namespaced_deployment(name=name, namespace=namespace)

    def create_deployment(self, v1_deployment, namespace):
        return self.get_apps_api().create_namespaced_deployment(namespace=namespace, body=v1_deployment)

    def remove_deployment(self, name, namespace):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        return self.get_apps_api().delete_namespaced_deployment(name=name, body=body, namespace=namespace)

    def replace_deployment(self, name, v1_deployment, namespace):
        return self.get_apps_api().replace_namespaced_deployment(namespace=namespace, body=v1_deployment, name=name)

    def is_deployment_existing(self, name, namespace):
        try:
            deployment = self.read_deployment(name=name, namespace=namespace)
            if deployment:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def read_statefulset(self, name, namespace):
        return self.get_apps_api().read_namespaced_stateful_set(name=name, namespace=namespace)

    def create_statefulset(self, v1_statefullset, namespace):
        return self.get_apps_api().create_namespaced_stateful_set(namespace=namespace, body=v1_statefullset)

    def remove_statefulset(self, name, namespace):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        return self.get_apps_api().delete_namespaced_stateful_set(name=name, body=body, namespace=namespace)

    def replace_statefulset(self, name, v1_statefullset, namespace):
        return self.get_apps_api().replace_namespaced_stateful_set(namespace=namespace, body=v1_statefullset, name=name)

    def is_statefulset_existing(self, name, namespace):
        try:
            statefulset = self.read_statefulset(name=name, namespace=namespace)
            if statefulset:
                return True
        except ApiException as e:
            if e.status == 404:
                return False
