#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import importlib


class K8SResourceFactory(object):
    def __init__(self, deployed):
        self.__deployed = deployed

    def validate_resource_types(self, resources):
        unsupported_resource_types = filter(lambda resource_type: resource_type not in self._get_supported_resource_types(),
                                       map(lambda resource: resource['kind'], resources))
        return not unsupported_resource_types, unsupported_resource_types

    @staticmethod
    def get_resource_order():
        return {
            'Pod': {'Create': 60, 'Modify': 50, 'Destroy': 40},
            'Service': {'Create': 67, 'Modify': 55, 'Destroy': 45},
            'Secret': {'Create': 67, 'Modify': 55, 'Destroy': 45},
            'Ingress': {'Create': 68, 'Modify': 68, 'Destroy': 32},
            'ConfigMap': {'Create': 67, 'Modify': 55, 'Destroy': 45},
            'Deployment': {'Create': 62, 'Modify': 53,'Destroy': 42},
            'StatefulSet': {'Create': 68, 'Modify': 56, 'Destroy': 43},
            'CronJob': {'Create': 68, 'Modify': 56, 'Destroy': 43},
            'PersistentVolume': {'Create': 56, 'Modify': 47, 'Destroy': 44},
            'PersistentVolumeClaim': {'Create': 58, 'Modify': 48, 'Destroy': 43},
            'StorageClass': {'Create': 54, 'Modify': 46, 'Destroy': 45}
        }

    @staticmethod
    def get_recreate_resources():
        return ['PersistentVolumeClaim']

    @staticmethod
    def get_resource_wait_details():
        return {
            "Create": {
                "Default": {'script': 'create_update_wait', 'action': "created"},
                "Pod": {'script': 'create_update_wait', 'action': "in running state"},
                "Deployment": {'script': 'deployment/create_wait', 'action': "in running state"},
                "StatefulSet": {'script': 'statefulset/create_wait', 'action': "in running state"},
                "PersistentVolumeClaim": {'script': 'create_update_wait', 'action': "in Bound phase"}
            },
            "Destroy": {
                "Default": {'script': 'delete_wait', 'action': "destroyed completely"}
            },
            "Modify": {
                "Default": {'script': 'create_update_wait', 'action': "modified"},
                "Pod": {'script': 'create_update_wait', 'action': "in running state"},
                "Deployment": {'script': 'deployment/update_wait', 'action': "in running state"},
                "StatefulSet": {'script': 'statefulset/update_wait', 'action': "in running state"}
            }
        }

    def get(self, data):
        return self._resolve(data)

    def _resolve(self, data):
        clazz = "{0}ResourceProvider".format(data["kind"])
        factory_module = importlib.import_module("xld.kubernetes.resource.provider")
        provider_clazz = getattr(factory_module, clazz)
        instance = provider_clazz(self.__deployed.container, data['apiVersion'] if 'apiVersion' in data else 'v1')
        return instance

    @staticmethod
    def _get_supported_resource_types():
        return K8SResourceFactory.get_resource_order().keys()
