
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import os
import base64
import string
import tempfile
import socket
import ssl
import urllib3

from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.eks.eks_helper import EKSHelper

from kubernetes.config.kube_config import KubeConfigLoader
from kubernetes.client.configuration import Configuration
from kubernetes.client import ApiClient

from com.xebialabs.deployit.plugin.kubernetes import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption

from kubernetes.client.rest import RESTClientObject
from certifi.core import where

setattr(socket, 'SOL_TCP', 6)


class KubernetesClientFactory(object):
    def __init__(self, container):
        self.__container = container
        self.__context = "kb_context"
        self.__cluster = "kb_cluster"
        self.__user = "kb_user"
        if hasattr(container, 'isEKS'):
            self.__eks_helper = EKSHelper(container.accessKey, container.accessSecret)

    def create(self):
        # Generate AWS k8s token if needed
        if self.__container.isEKS:
            token = self.__eks_helper.get_k8s_token(self.__container.clusterName, self.__container.regionName) if self.__container.regionName else self.__eks_helper.get_k8s_token(self.__container.clusterName)
        else:
            token = self.__container.token

        cluster = self._create_cluster(server_url=self.__container.apiServerURL, skip_tls=self.__container.skipTLS)
        user = self._create_user(token=token, username=self.__container.username, password=self.__container.password)
        context = self._create_context()
        config_params = self._get_config_params(cluster=cluster, context=context, user=user)
        proxy_headers = self.create_proxy_headers(self.__container)
        return KubernetesApiClient(configuration=self.__create_config(config_params), proxy_headers=proxy_headers)

    def _create_cluster(self, server_url=None, skip_tls=True):
        cluster = {"server": server_url,
                   'insecure-skip-tls-verify': "true" if skip_tls else None,
                   "certificate-authority-data": self._ensure_b64encoded(self.__container.caCert) if self.__container.caCert else None}
        return CommonUtils.remove_none_keys(cluster)

    def _create_context(self):
        return {"cluster": self.__cluster, "user": self.__user}

    def _create_user(self, token=None, username=None, password=None):
        user = {
            "client-certificate-data": self._ensure_b64encoded(self.__container.tlsCert) if self.__container.tlsCert else None,
            "client-key-data": self._ensure_b64encoded(self.__container.tlsPrivateKey) if self.__container.tlsPrivateKey else None,
            "token": token, "username": username,
            "password": password}
        return CommonUtils.remove_none_keys(user)

    def _ensure_b64encoded(self, data):
        if "-----BEGIN" in data and "-----END" in data:
            return base64.b64encode(self._correct(data))
        else:
            return data.strip()

    def _get_config_params(self, context, cluster, user):
        return {"current-context": self.__context,
                "contexts": [{"name": self.__context, "context": context}],
                "clusters": [{"name": self.__cluster, "cluster": cluster}],
                "users": [{"name": self.__user, "user": user}]}

    def __create_config(self, config_params):
        client_config = type.__call__(Configuration)
        config_loader = KubeConfigLoader(config_dict=config_params, active_context=self.__context)
        config_loader.load_and_set(client_config)
        if self.__container.skipTLS:
            self._disable_https_warnings()
            client_config.ssl_ca_cert = self._set_ca_bundle_path(path="certifi/cacert.pem")
        client_config.proxy = self.create_proxy_url(self.__container)
        return client_config

    @staticmethod
    def _disable_https_warnings():
        import urllib3
        urllib3.disable_warnings()

    @staticmethod
    def _correct(cert):
        if "-----" in cert and " " in cert.split('-----')[2]:
            return KubernetesClientFactory._cert_replace_spaces(cert).strip()
        else:
            return cert.strip()

    @staticmethod
    def _cert_replace_spaces(cert):
        splits=cert.split('-----')
        splits[2]=string.replace(splits[2], ' ', '\n')
        return '-----'.join(splits)

    def _set_ca_bundle_path(self, path):
        ca_bundle_path = self.__extract_file_from_jar(path)
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path
        return ca_bundle_path

    def __extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
            return tmp_abs_path
        else:
            return None

    @staticmethod
    def create_proxy_headers(container):
        proxies = None
        if container.proxyServer:
            proxies = urllib3.make_headers(proxy_basic_auth="{}:{}".format(container.proxyServer.username, container.proxyServer.password))

        return proxies

    @staticmethod
    def create_proxy_url(container):
        proxy = None
        if container.proxyServer:
            proxy = "{}://{}:{}".format(str(container.proxyServer.protocol).lower(), container.proxyServer.hostname, container.proxyServer.port)
        return proxy


class KubernetesApiClient(ApiClient):
    def __init__(self, configuration=None, header_name=None, header_value=None, cookie=None, proxy_headers=None):
        if configuration is None:
            configuration = Configuration()
        self.configuration = configuration

        self.proxy_headers = proxy_headers

        self.rest_client = KubernetesRESTClientObject(configuration, proxy_headers=proxy_headers)
        self.default_headers = {}
        if header_name is not None:
            self.default_headers[header_name] = header_value
        self.cookie = cookie
        # Set default User-Agent.
        self.user_agent = 'Swagger-Codegen/10.0.1/python'


class KubernetesRESTClientObject(RESTClientObject):
    def __init__(self, configuration, pools_size=4, maxsize=None, proxy_headers=None):
        if configuration.verify_ssl:
            cert_reqs = ssl.CERT_REQUIRED
        else:
            cert_reqs = ssl.CERT_NONE

        self.proxy_headers = proxy_headers

        # ca_certs
        if configuration.ssl_ca_cert:
            ca_certs = configuration.ssl_ca_cert
        else:
            # if not set certificate file, use Mozilla's root certificates.
            ca_certs = where()

        addition_pool_args = {}
        if configuration.assert_hostname is not None:
            addition_pool_args['assert_hostname'] = configuration.assert_hostname

        if maxsize is None:
            if configuration.connection_pool_maxsize is not None:
                maxsize = configuration.connection_pool_maxsize
            else:
                maxsize = 4

        # https pool manager
        if configuration.proxy:
            self.pool_manager = urllib3.ProxyManager(
                num_pools=pools_size,
                maxsize=maxsize,
                cert_reqs=cert_reqs,
                ca_certs=ca_certs,
                cert_file=configuration.cert_file,
                key_file=configuration.key_file,
                proxy_url=configuration.proxy,
                proxy_headers=proxy_headers,
                **addition_pool_args
            )
        else:
            self.pool_manager = urllib3.PoolManager(
                num_pools=pools_size,
                maxsize=maxsize,
                cert_reqs=cert_reqs,
                ca_certs=ca_certs,
                cert_file=configuration.cert_file,
                key_file=configuration.key_file,
                **addition_pool_args
            )
