from kubernetes.client import V1DeleteOptions
from openshift.dynamic import DynamicClient
from xld.kubernetes.resource.provider import Resource
from xld.kubernetes import client
from xld.openshift.connector import OpenShiftConnector
from abc import abstractmethod
import time


class OpenShiftResource(Resource):
    def __init__(self, container, api_version='v1'):
        super(OpenShiftResource, self).__init__(container, api_version)
        connector = OpenShiftConnector(container.container)
        self.oapi = connector.oapi
        self.apiclient = connector.apiclient

    def get_api_client(self, kind):
        print ("[Using API version: {}]".format(self.api_version))
        dynamic_client = DynamicClient(self.oapi.client)
        if kind.lower() == "template":
            return dynamic_client.resources.get(api_version=self.api_version, kind=kind, name="templates")
        else:
            return dynamic_client.resources.get(api_version=self.api_version, kind=kind)

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        return self.apiclient.rest_client.pool_manager.__exit__(exc_type, exc_val, exc_tb)


class OpenshiftResourceProvider(OpenShiftResource):
    def __init__(self, container, kind, api_version):
        super(OpenshiftResourceProvider, self).__init__(container, api_version)
        self.kind=kind
        self.api_client = self.get_api_client(self.kind)
        self.patch_types = {
            "json": "application/json-patch+json",
            "merge": "application/merge-patch+json",
            "strategic": "application/strategic-merge-patch+json"
        }

    def create(self, namespace, resource_definition):
        return self.api_client.create(body=resource_definition, namespace=namespace)

    def modify(self, namespace, resource_definition, patch_type='strategic', update_method='patch'):
        if update_method == 'patch':
            if patch_type not in self.patch_types:
                raise RuntimeError("Merge patch type '{}' not supported. Please use 'json', 'merge' or 'strategic'.".format(patch_type))
            return self.api_client.patch(body=resource_definition, namespace=namespace, content_type=self.patch_types[patch_type])
        elif update_method == 'apply':
            return self.api_client.replace(body=resource_definition, namespace=namespace)
        elif update_method == 'replace':
            resource_name = resource_definition["metadata"]["name"]
            print("Deleting '{}'.".format(resource_name))
            self.api_client.delete(name=resource_name, namespace=namespace, body=client.V1DeleteOptions(
                propagation_policy='Foreground'
            ))
            self.wait_until_deleted(namespace, resource_definition)
            print("Recreating '{}'.".format(resource_name))
            return self.api_client.create(body=resource_definition, namespace=namespace)
        else:
            raise Exception("Update Method '{}' not supported.".format(update_method))


    def delete(self, namespace, resource_definition, propagation_policy='Foreground'):
        return self.api_client.delete(name=resource_definition["metadata"]["name"], namespace=namespace, body=client.V1DeleteOptions(
            propagation_policy=propagation_policy
        ))

    def filter_resources_by_definition(self, namespace, resource_definition):
        return self.api_client.get(namespace=namespace, field_selector="metadata.name={}".format(
            resource_definition["metadata"]["name"]))

    def wait_until_deleted(self, namespace, resource_definition):
        exists = True
        resource_name = resource_definition["metadata"]["name"]
        while exists:
            time.sleep(5)
            print("Checking for existence of '{}'.".format(resource_name))
            resources = self.filter_resources_by_definition(namespace=namespace, resource_definition=resource_definition)
            if 'items' not in dir(resources) or not bool(resources.items):
                exists = False
                print("Resource '{}' deleted.".format(resource_name))
            else:
                print("Resource '{}' still exists.".format(resource_name))


class RouteResourceProvider(OpenShiftResource):
    route_api_version="route.openshift.io/v1"
    kind="Route"

    def create(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.route_api_version, kind=self.kind).create(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.route_api_version, kind=self.kind).patch(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.resources.get(api_version=self.route_api_version, kind=self.kind).delete(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.resources.get(api_version=self.route_api_version, kind=self.kind).get(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ImageStreamResourceProvider(OpenShiftResource):
    image_api_version="image.openshift.io/v1"
    kind="ImageStream"

    def create(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.image_api_version, kind=self.kind).create(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.image_api_version, kind=self.kind).patch(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.resources.get(api_version=self.image_api_version, kind=self.kind).delete(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.resources.get(api_version=self.image_api_version, kind=self.kind).get(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class BuildConfigResourceProvider(OpenShiftResource):
    api_version="config.openshift.io/v1"
    kind="Build"

    def create(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).create(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).patch(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).delete(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).get(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class DeploymentConfigResourceProvider(OpenShiftResource):
    api_version="v1"
    kind="DeploymentConfig"

    def create(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).create(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).patch(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).delete(
            namespace=namespace,
            name=resource_definition["metadata"]["name"]
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.oapi.resources.get(api_version=self.api_version, kind=self.kind).get(
            namespace=namespace,
            field_selector=field_selector
        )
        return response
