package com.xebialabs.xldeploy.provisioner.api

import java.util
import javax.xml.bind.DatatypeConverter

import com.xebialabs.deployit.engine.api.execution.{StepState, TaskPreviewBlock}
import com.xebialabs.deployit.engine.tasker.TaskSpecification
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xldeploy.provisioner._
import com.xebialabs.xldeploy.provisioner.api.JavaCollectionJsFormat._
import com.xebialabs.xltype.serialization.json.{StepStateJsonConverter, TaskPreviewBlockJsonConverter}
import spray.json.DefaultJsonProtocol._
import spray.json._

import scala.collection.convert.wrapAll._
import scala.util.Try


trait JsonApiFormat {

  implicit val provisionedEnvironmentFormat = configurationItemFormat[ProvisionedBlueprint]

  implicit val ciFormat = configurationItemFormat[ConfigurationItem]

  implicit val validationErrorFormat = jsonFormat3(ValidationError.apply)

  implicit val provisioningRequestFormat = jsonFormat3(ProvisioningRequest.apply)

  implicit val provisioningFormat = jsonFormat3(Provision.apply)

  implicit val deprovisioningRequestFormat = jsonFormat1(DeprovisioningRequest.apply)

  implicit val provisioningTaskFormat = jsonFormat2(ProvisioningTask.apply)

  implicit val taskSpecificationFormat = new RootJsonFormat[TaskSpecification] {
    override def read(json: JsValue): TaskSpecification = deserializationError("TaskSpecification cannot be parsed")

    override def write(obj: TaskSpecification): JsValue = {
      JsObject(Map("taskId" -> JsString(obj.getId)))
    }
  }

  implicit val taskPreviewBlockFormat = new RootJsonFormat[TaskPreviewBlock] {
    override def read(json: JsValue): TaskPreviewBlock = deserializationError("TaskPreviewBlock cannot be parsed")

    override def write(obj: TaskPreviewBlock): JsValue = {
      new TaskPreviewBlockJsonConverter().write(obj).parseJson
    }
  }

  implicit val stepStateFormat = new RootJsonFormat[StepState] {
    override def read(json: JsValue): StepState = deserializationError("StepState cannot be parsed")

    override def write(obj: StepState): JsValue = {
      new StepStateJsonConverter().toJson(obj).parseJson
    }
  }

  implicit val taskPreviewFormat = jsonFormat2(TaskPreview.apply)

  implicit val stepPreviewFormat = jsonFormat2(StepPreview.apply)

  implicit val selectedProvisionFormat = jsonFormat2(SelectedProvision.apply)

  def configurationItemFormat[CI <: ConfigurationItem]: RootJsonFormat[CI] = new RootJsonFormat[CI] {

    override def write(ci: CI): JsValue = {
      val descriptors: util.Collection[PropertyDescriptor] = ci.getType.getDescriptor.getPropertyDescriptors

      val jsonSyntheticFields = descriptors.collect {
        case pd if pd.get(ci).nonEmpty =>
          val property = pd.get(ci)
          val format = jsFormatByPropertyDescriptor(pd).asInstanceOf[JsonFormat[AnyRef]]
          pd.getName -> property.toJson(format)
      }.toSeq

      val jsonFields = jsonSyntheticFields :+ "id" -> JsString(ci.getId) :+ "type" -> JsString(ci.getType.toString)

      JsObject(jsonFields: _*)
    }

    override def read(json: JsValue): CI = json match {
      case CiWithIdAndType(ci, fields) =>
        fields.foreach { case (k, v) =>
          val pd = ci.getType.getDescriptor.getPropertyDescriptor(k)
          val jsFormat = jsFormatByPropertyDescriptor(pd)
          ci.setProperty(pd.getName, jsFormat.read(v))
        }
        ci.asInstanceOf[CI]
      case _ =>
        deserializationError("Cannot parse ci")
    }
  }

  private def jsFormatByPropertyDescriptor(propertyDescriptor: PropertyDescriptor): JsonFormat[_] = propertyDescriptor.getKind match {
    case PropertyKind.BOOLEAN => BooleanJsonFormat
    case PropertyKind.INTEGER => IntJsonFormat
    case PropertyKind.DATE => DateJsonFormat
    case PropertyKind.ENUM => StringJsonFormat
    case PropertyKind.CI => jsFormatByCiReference(propertyDescriptor.getReferencedType)
    case PropertyKind.LIST_OF_STRING => javaListFormat(StringJsonFormat)
    case PropertyKind.LIST_OF_CI => javaListFormat(jsFormatByCiReference(propertyDescriptor.getReferencedType))
    case PropertyKind.MAP_STRING_STRING => javaMapFormat(StringJsonFormat, StringJsonFormat)
    case PropertyKind.SET_OF_STRING => javaSetFormat(StringJsonFormat)
    case PropertyKind.SET_OF_CI => javaSetFormat(jsFormatByCiReference(propertyDescriptor.getReferencedType))
    case PropertyKind.STRING => StringJsonFormat
  }

  private def jsFormatByCiReference(referenceType: Type) = referenceType match {
    case EmbeddedDeployed() => embeddedDeployedFormat
    case _ => CiReferenceFormat
  }

  private val embeddedDeployedFormat = configurationItemFormat[EmbeddedDeployedType]

  private object DateJsonFormat extends JsonFormat[util.Date] {

    override def write(obj: util.Date): JsValue = {
      val calendar = util.Calendar.getInstance()
      calendar.setTime(obj)
      JsString(DatatypeConverter.printDateTime(calendar))
    }

    override def read(json: JsValue): util.Date = json match {
      case JsString(date) => Try(DatatypeConverter.parseDateTime(date).getTime).getOrElse(deserializationError("Cannot parse date"))
      case _ => deserializationError("Cannot parse date")
    }
  }

  private object CiReferenceFormat extends JsonFormat[ConfigurationItem] {

    override def read(json: JsValue): ConfigurationItem = json match {
      case CiWithIdAndType(ci, _) => ci
      case _ => deserializationError("Cannot parse ci reference because ref or type are not present")
    }

    override def write(obj: ConfigurationItem): JsValue =
      JsObject("id" -> JsString(obj.getId), "type" -> JsString(obj.getType.toString))
  }

  private object CiWithIdAndType {
    def unapply(jsonValue: JsValue): Option[(ConfigurationItem, Seq[JsField])] = jsonValue match {
      case obj@JsObject(fields) =>
        def field(fieldName: String): Option[String] = {
          fields.get(fieldName) match {
            case Some(JsString(value)) => Some(value)
            case _ => None
          }
        }
        for (ciId <- field("id"); ciType <- field("type"))
          yield (createCi(ciType, ciId), obj.fields.filterKeys(name => name != "id" && name != "type").toSeq)
      case _ =>
        None
    }
  }

  private val EmbeddedDeployed = new CiAssignableTo(classOf[EmbeddedDeployedType])

  private class CiAssignableTo(classs: Class[_]) {
    val ciType = Type.valueOf(classs)

    def unapply(matchType: Type): Boolean = ciType.isSuperTypeOf(matchType) || ciType == matchType
  }

}

object JsonApiFormat extends JsonApiFormat
