package com.xebialabs.xldeploy.provisioner

import java.security.SecureRandom

import com.xebialabs.deployit.plugin.api.udm.{ConfigurationItem, Environment}
import com.xebialabs.deployit.security.permission.{Permission, PlatformPermissions}
import org.hashids.Hashids

package object api {

  private val random = new SecureRandom()

  implicit class ToOptionUtil[T <: Traversable[_]](val value: T) extends AnyVal {
    def toOption: Option[T] = {
      if (value.isEmpty)
        None
      else
        Option(value)
    }
  }

  implicit class ProvisionedBlueprintUtils(val provisionedBlueprint: ProvisionedBlueprint) extends AnyVal {
    def withEnvironment(environment: Option[Environment]): ProvisionedBlueprint = {
      environment.foreach(provisionedBlueprint.setProvisionedEnvironment)
      provisionedBlueprint
    }
  }

  def hashId(salt: String) = {
    val hashids = new Hashids(salt)
    hashids.encode(getNextRandomBytes(3): _*)
  }

  private def getNextRandomBytes(n: Int) = {
    val bytes = new Array[Byte](n)
    random.nextBytes(bytes)
    bytes.map(b => Math.abs(b.toLong))
  }

  sealed trait PermissionChecker[CI <: ConfigurationItem] {
    val permission: Permission

    def checkPermission(ci: CI) = {
      require(permission.getPermissionHandler.hasPermission(ci.getId), s"User does not have permission ${permission.getPermissionName} on ${ci.getId}")
    }
  }

  object Implicits {
    implicit object ReadProvisioningPackage extends PermissionChecker[ProvisioningPackage] {
      override val permission: Permission = PlatformPermissions.READ
    }

    implicit object EditProvisioningEnvironment extends PermissionChecker[ProvisioningEnvironment] {
      override val permission: Permission = PlatformPermissions.EDIT_REPO
    }

    implicit object EditProvisionedBlueprint extends PermissionChecker[ProvisionedBlueprint] {
      override val permission: Permission = PlatformPermissions.EDIT_REPO
    }
  }
}
