package com.xebialabs.xldeploy.provisioner
package steps

import com.xebialabs.deployit.plugin.api.flow.{PreviewStep, ExecutionContext, Preview, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.ChangeSet
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._

@StepMetadata(name = "delete-all-provisioned-cis")
class DeleteAllProvisionedItems extends ProvisionedItemsStep with PreviewStep with Logging {

  override def validScope: Scope = Scope.PRE_PLAN

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext) {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = s"Deleting all provisioned configuration items"
    }
    order = if (order != null) order else 25
    setParameters(ctx)
  }

  override def getPreview: Preview = {
    Preview.withContents(s"Deleting following configuration items:\n${allCiToDelete.mkString("\n")}")
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput("Starting deletion of provisioned items")
    val changeSet = new ChangeSet
    val ciIds = allCiToDelete
    ctx.logOutputRaw(s"Deleting following configuration items:\n${ciIds.mkString("\n")}")
    changeSet.delete(ciIds.toList)
    resetProvisionedEnvironment(changeSet)
    resetProvisioneds(changeSet)
    repositoryService.execute(changeSet)
    ctx.logOutput("Deletion of provisioned items is done")
    StepExitCode.SUCCESS
  }

  private def allCiToDelete: Set[String] = {
    resolvedTemplateOnProvisionedBlueprint ++ resolvedTemplatesOnProvisioneds + provisionedBlueprint.getProvisionedEnvironment.getId
  }

  private def resolvedTemplateOnProvisionedBlueprint: Set[String] = {
    provisionedBlueprint.getBoundConfigurationItems.map(_.getId).toSet
  }

  private def resolvedTemplatesOnProvisioneds: Set[String] = {
    provisioneds.flatMap(_.boundConfigurationItems.toSet).map(_.getId)
  }

  private def resetProvisioneds(changeSet: ChangeSet): Unit = {
    val provisionedWithBoundCIs = provisioneds.filterNot(_.boundConfigurationItems.isEmpty)
    provisionedWithBoundCIs.foreach(_.boundConfigurationItems = new JHashSet[ConfigurationItem])
    changeSet.update(provisionedWithBoundCIs.toList)
  }

  private def resetProvisionedEnvironment(changeSet: ChangeSet): Unit = {
    if(repositoryService.exists(provisionedBlueprint.getId)) {
      provisionedBlueprint.setProvisionedEnvironment(null)
      provisionedBlueprint.setBoundConfigurationItems(new JHashSet[ConfigurationItem]())
      changeSet.update(provisionedBlueprint)
    }
  }
}
