#
# Copyright (c) 2021. All rights reserved.
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

"""
Module containing class for common endpoint implementations across all TFE Endpoints.
"""

import json
import logging
import os
import zipfile
import tempfile
import requests
from org.apache.http import HttpHost
from org.apache.commons.io import IOUtils
from java.nio.charset import Charset

from org.apache.http.impl.client import HttpClientBuilder
from org.apache.http.client.methods import HttpGet
from org.apache.http.ssl import SSLContextBuilder
from org.apache.http.conn.ssl import TrustSelfSignedStrategy
from org.apache.http.conn.ssl import NoopHostnameVerifier


class TFEEndpoint(object):
    """
    Base class providing common CRUD operation implementations across all TFE Endpoints.
    """

    def __init__(self, base_url, organization, headers):
        self._base_url = base_url
        self._headers = headers
        self._organization = organization
        self._logger = logging.getLogger(self.__class__.__name__)
        self._logger.setLevel(logging.INFO)
        if self._organization.verifyCertificates:
            path_to_cert = self.local_extract_zipped_paths(self._organization.pathToCAFile)
            self._logger.debug("CA File:" + path_to_cert)
            self._verify = path_to_cert
        else:
            self._verify = False

        if self._organization.proxyServer is not None:
            proxy_server = self._organization.proxyServer
            proxy_url = "{}://{}:{}".format(str(proxy_server.protocol).lower(), proxy_server.hostname, proxy_server.port)
            self._proxies = {'http': proxy_url, 'https': proxy_url}
            self._java_proxy = HttpHost(proxy_server.hostname, proxy_server.port, str(proxy_server.protocol).lower())
        else:
            self._proxies = None
            self._java_proxy = None

        if self._organization.organizationName is not None:
            self._organization_name = self._organization.organizationName
        else:
            self._organization_name = self._organization.name

    def _create(self, url, payload):
        """
        Implementation the common create resource pattern for the TFE API.
        """
        results = None
        self._logger.debug(json.dumps(payload))
        req = requests.post(url, json.dumps(payload), headers=self._headers, verify=self._verify, proxies=self._proxies)

        if req.status_code == 201:
            results = json.loads(req.content)
            return results
        else:
            err = json.loads(req.content.decode("utf-8"))
            self._logger.error(err)
            return err

    def _destroy(self, url):
        """
        Implementation of the common destroy resource pattern for the TFE API.
        """
        req = requests.delete(url, headers=self._headers, verify=self._verify, proxies=self._proxies)

        valid_status_codes = [200, 204]
        if req.status_code in valid_status_codes:
            self._logger.debug("Terraform Enterprise resource at URL [{url}] destroyed.".format(url=url))
        else:
            err = json.loads(req.content.decode("utf-8"))
            self._logger.error(err)

    def _ls(self, url):
        """
        Implementation of the common list resources pattern for the TFE API.
        """
        results = None
        req = requests.get(url, headers=self._headers, verify=self._verify, proxies=self._proxies)

        if req.status_code == 200:
            results = json.loads(req.content)
        else:
            err = json.loads(req.content.decode("utf-8"))
            self._logger.error(err)

        return results

    def _show(self, url):
        """
        Implementation of the common show resource pattern for the TFE API.
        """
        req = requests.get(url, headers=self._headers, verify=self._verify, proxies=self._proxies)

        if req.status_code == 200:
            results = json.loads(req.content)
            return results
        else:
            err = json.loads(req.content.decode("utf-8"))
            self._logger.error(err)
            return err

    def _update(self, url, payload):
        """
        Implementation of the common update resource pattern for the TFE API.
        """
        req = requests.patch(url, data=json.dumps(payload), headers=self._headers, verify=self._verify, proxies=self._proxies)

        if req.status_code == 200:
            results = json.loads(req.content)
        else:
            err = json.loads(req.content.decode("utf-8"))
            self._logger.error(err)

        return results

    def _download(self, url):
        self._logger.debug("_download {0}".format(url))

        http_client_builder = HttpClientBuilder.create()
        http_client_builder.setProxy(self._java_proxy)

        ssl_context = SSLContextBuilder().loadTrustMaterial(None, TrustSelfSignedStrategy()).build()
        http_client_builder.setSSLContext(ssl_context).setSSLHostnameVerifier(NoopHostnameVerifier())

        client = http_client_builder.build()
        http_response = client.execute(HttpGet(url))
        content = IOUtils.toString(http_response.getEntity().getContent(), Charset.forName("UTF-8"))
        return content

    def _stream(self, url):
        """
        Stream the resource
        """
        results = None
        self._logger.debug("_stream {0}".format(url))
        req = requests.get(url, headers=self._headers, verify=self._verify, proxies=self._proxies, stream=True)
        return req

    def format_error_message(self, run):
        if run is None:
            return "None"

        if 'errors' in run:
            return ",".join(['{title} ({status})'.format(**error) for error in run['errors']])
        else:
            if run is "None":
                "Empty Error Message"
            else:
                return run

    def local_extract_zipped_paths(self, path):
        """Replace nonexistent paths that look like they refer to a member of a zip
        archive with the location of an extracted copy of the target, or else
        just return the provided path unchanged.
        """
        if os.path.exists(path):
            # this is already a valid path, no need to do anything further
            return path

        # find the first valid part of the provided path and treat that as a zip archive
        # assume the rest of the path is the name of a member in the archive
        archive, member = os.path.split(path)
        while archive and not os.path.exists(archive):
            archive, prefix = os.path.split(archive)
            member = '/'.join([prefix, member])

        if not zipfile.is_zipfile(archive):
            return path

        zip_file = zipfile.ZipFile(archive)
        if member not in zip_file.namelist():
            return path

        # we have a valid zip archive and a valid member of that archive
        tmp = tempfile.gettempdir()
        extracted_path = os.path.join(tmp, *member.split('/'))
        if not os.path.exists(extracted_path):
            extracted_path = zip_file.extract(member, path=tmp)

        return extracted_path
