package com.xebialabs.deployit.plugin.xlrelease;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.generic.ci.BaseGenericContainer;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.ci.NestedContainer;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.generic.step.WaitStep;

import static com.xebialabs.deployit.plugin.xlrelease.RecreateDataContributor.INSTALLATION;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;
import static java.lang.String.format;

public class ServerRestartContributor {

    @Contributor
    public void restartContainers(Deltas deltas, DeploymentPlanningContext plan) {
        for (BaseGenericContainer container : getStartableContainers(deltas.getDeltas())) {
            addStartSteps(plan, container);
        }
        for (BaseGenericContainer container : getStoppableContainers(deltas.getDeltas())) {
            addStopSteps(plan, container);
        }
    }

    private static void addStartSteps(DeploymentPlanningContext plan, BaseGenericContainer target) {
        checkArgument(!isNullOrEmpty(target.getStartScript()), format("%s start script must be specified when no restart script defined.", target.getId()));

        plan.addStep(createScriptStep("Start", target, target.getStartScript(), target.getStartOrder(), target.getStartClasspathResources(), target.getStartTemplateClasspathResources()));
        addWaitStep(plan, "start", target, target.getStartWaitTime(), target.getStartOrder());
    }

    private static void addStopSteps(DeploymentPlanningContext plan, BaseGenericContainer target) {
        checkArgument(!isNullOrEmpty(target.getStopScript()), format("%s stop script must be specified when no restart script defined.", target.getId()));

        plan.addStep(createScriptStep("Stop", target, target.getStopScript(), target.getStopOrder(), target.getStopClasspathResources(),target.getStopTemplateClasspathResources()));
        addWaitStep(plan, "stop", target, target.getStopWaitTime(), target.getStopOrder());
    }

    private static void addWaitStep(DeploymentPlanningContext plan, String action, BaseGenericContainer target, int waitTime, int scriptOrder) {
        if (waitTime > 0) {
            plan.addStep(new WaitStep(scriptOrder + 1, waitTime, target.getName(), action));
        }
    }

    private static Step createScriptStep(String verb, BaseGenericContainer container, String script, int scriptOrder, Set<String> classpathResources, Set<String> templateClasspathResources) {
        Map<String, Object> vars = Maps.newHashMap();
        vars.put("container", container);
        ScriptExecutionStep step = new ScriptExecutionStep(scriptOrder, script, container, vars, getDescription(verb, container));
        step.setClasspathResources(newArrayList(classpathResources));
        step.setTemplateClasspathResources(newArrayList(templateClasspathResources));
        return step;
    }



    private static String getDescription(String verb, BaseGenericContainer container) {
        return format("%s %s", verb, container.getName());
    }


    private static Set<BaseGenericContainer> getStartableContainers(List<Delta> operations) {
        final Set<BaseGenericContainer> targets = Sets.newTreeSet();
        for (Delta operation : operations) {
            if (shouldStartServer(operation)) {
                targets.add(getContainerToRestart(operation.getDeployed().getContainer()));
            }
        }
        return targets;
    }

    private static Set<BaseGenericContainer> getStoppableContainers(List<Delta> operations) {
        final Set<BaseGenericContainer> targets = Sets.newTreeSet();
        for (Delta operation : operations) {
            if (shouldStopServer(operation)) {
                targets.add(getContainerToRestart(operation.getPrevious().getContainer()));
            }
        }
        return targets;
    }

    private static BaseGenericContainer getContainerToRestart(com.xebialabs.deployit.plugin.api.udm.Container c) {
        if (c instanceof Container) {
            return (Container) c;
        } else if (c instanceof NestedContainer) {
            NestedContainer nc = (NestedContainer) c;
            if(!isNullOrEmpty(nc.getStartScript()) || !isNullOrEmpty(nc.getStopScript()) || !isNullOrEmpty(nc.getRestartScript())) {
                return nc;
            } else {
                return getContainerToRestart(nc.getParentContainer());
            }
        } else {
            throw new IllegalStateException(format("Container [%s] is not a generic (nested) container", c));
        }
    }

    private static boolean shouldStartServer(Delta delta) {
        if (delta.getDeployed() == null) {
            return false;
        }
        return delta.getDeployed().getType().equals(INSTALLATION) && (delta.getOperation() == CREATE || delta.getOperation() == MODIFY);
    }

    private static boolean shouldStopServer(Delta delta) {
        if (delta.getPrevious() == null) {
            return false;
        }
        return delta.getPrevious().getType().equals(INSTALLATION) && (delta.getOperation() == DESTROY || delta.getOperation() == MODIFY);
    }

    private static boolean isNullOrEmpty(String s) {
        return nullToEmpty(s).trim().isEmpty();
    }


}
