package com.xebialabs.deployit.provision
package resolver

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployable
import com.xebialabs.deployit.provision.resolver.placeholder.PlaceholderResolver
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, Dictionaries}

import scala.collection.convert.ImplicitConversions._
import scala.collection.mutable
import scala.language.existentials
import scala.util.{Failure, Success, Try}

object TemplateResolver {

  def apply(provisionedBlueprint: DeployedApplication): TemplateResolver = apply(provisionedBlueprint, ignoreContextPlaceholders = false)

  def apply(provisionedBlueprint: DeployedApplication, ignoreContextPlaceholders: Boolean): TemplateResolver = apply(Dictionaries.of(provisionedBlueprint.getEnvironment)
    .withAdditionalEntries(provisionedBlueprint.getUnresolvedPlaceholdersWithValues).consolidate(), provisionedBlueprint, ignoreContextPlaceholders)

  def apply(dictionary: ConsolidatedDictionary, provisionedBlueprint: DeployedApplication, ignoreContextPlaceholders: Boolean = false): TemplateResolver =
    new TemplateResolver(dictionary, provisionedBlueprint, ignoreContextPlaceholders)

  private[resolver] type ResolvedCache = mutable.Map[String, Try[ConfigurationItem]]

  private[resolver] type ParentCiId = String

  private[resolver] type TemplateName = String

  private[resolver] def emptyCache = new mutable.HashMap[String, Try[ConfigurationItem]]()

  type CiIdGenerator = (ParentCiId, TemplateName) => String

  case class ResolutionResult(templateId: String, instanceId: Try[String], configurationItem: Try[ConfigurationItem])

}

class TemplateResolver(dictionary: ConsolidatedDictionary,
                       provisionedBlueprint: DeployedApplication,
                       ignoreContextPlaceholders: Boolean = false) {

  import TemplateResolver._

  private lazy val idGenerator: TemplateResolver.CiIdGenerator = (parent, name) =>
    s"$parent/$name"

  private lazy val idGeneratorWithOrdinal: TemplateResolver.CiIdGenerator = (parent, name) =>
    s"$parent/$name-{{%ordinal%}}"

  private lazy val simpleIdGenerator: CiIdGenerator = (parent, name) => s"$parent/$name"

  private def pickCorrectIdGenerator(provisioned: Option[DeployedType] = None): CiIdGenerator =
    if (provisioned.withFilter(_.isInstanceOf[ProvisionedType]).map(_.asInstanceOf[ProvisionedType]).exists(_.getOrdinal > 1))
      idGeneratorWithOrdinal
    else
      idGenerator

  def resolveEnvironmentId(): Try[String] = {
    implicit val resolver: PlaceholderResolver = placeholderResolver()
    Try(resolveValue(rootDirectory("Environments") + "/" + provisionedBlueprint.getEnvironment.getName))
  }

  private def placeholderResolver(provisioned: Option[DeployedType] = None): PlaceholderResolver = PlaceholderResolver(dictionary, provisioned)

  def resolveBoundTemplate(provisioned: DeployedType): Set[ResolutionResult] = {
    val maybeProvisioned = Option(provisioned)
    implicit val resolver: PlaceholderResolver = placeholderResolver(maybeProvisioned)
    resolveRootTemplates(provisioned.getDeployable.asInstanceOf[BaseDeployable].getBoundTemplates.toSet, pickCorrectIdGenerator(maybeProvisioned))
  }

  def resolveBoundTemplate(deploymentPackage: DeploymentPackage): Set[ResolutionResult] = {
    implicit val resolver: PlaceholderResolver = placeholderResolver()
    resolveRootTemplates(deploymentPackage.getBoundTemplates.toSet, pickCorrectIdGenerator())
  }

  def resolveSingleRootTemplate(template: Template, provisioned: DeployedType): ResolutionResult = {
    val maybeProvisioned = Option(provisioned)
    implicit val alreadyResolvedTemplates: mutable.HashMap[String, Try[ConfigurationItem]] = emptyCache
    implicit val resolver: PlaceholderResolver = placeholderResolver(maybeProvisioned)
    resolveTemplateToCi(template, rootDirectory(template.getType.getDescriptor.getRootName), pickCorrectIdGenerator(maybeProvisioned))
  }

  private def resolveRootTemplates(templates: Set[Template], idGenerator: CiIdGenerator)(implicit resolver: PlaceholderResolver): Set[ResolutionResult] = {
    implicit val alreadyResolvedTemplates: mutable.HashMap[String, Try[ConfigurationItem]] = emptyCache
    val configurationItems = templates.flatMap(resolveRootTemplate(_, idGenerator))
    templates.foreach(setResolvedReferences)
    configurationItems
  }

  private def resolveRootTemplate(template: Template, idGenerator: CiIdGenerator)
                                 (implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Set[ResolutionResult] = {
    val instanceTypeDescriptor = instanceType(template).getDescriptor
    if (instanceTypeDescriptor.getRoot != ConfigurationItemRoot.NESTED) {
      val resolvedConfigurationItems = resolveTemplateHierarchy(template, rootDirectory(instanceTypeDescriptor.getRootName), idGenerator)
      resolvedConfigurationItems
    } else {
      throw new IllegalArgumentException(
        s"Template '${template.getId}' cannot be initialized under the package. Please create it under the correct parent template.")
    }
  }

  private def rootDirectory(rootName: String) = {
    val pathWithRoot = rootName + "/" + Option(provisionedBlueprint.getEnvironment.getDirectoryPath).getOrElse("")
    pathWithRoot.split("/").filterNot(_.isEmpty).mkString("/")
  }

  private def resolveTemplateHierarchy(template: Template, parentId: ParentCiId, idGenerator: CiIdGenerator)
                                      (implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Set[ResolutionResult] = {
    val resolutionResult = resolveTemplateToCi(template, parentId, idGenerator)
    resolveTemplateChildren(template, resolutionResult.instanceId, simpleIdGenerator) + resolutionResult
  }

  private def resolveTemplateChildren(template: Template, instanceId: Try[String], idGenerator: CiIdGenerator)
                                     (implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Set[ResolutionResult] = {
    instanceId match {
      case Success(iid) => template.childTemplates.flatMap(template => resolveTemplateHierarchy(template, iid, idGenerator)).toSet
      case Failure(_) => Set.empty
    }
  }

  private def resolveTemplateToCi(template: Template, parentId: String, idGenerator: CiIdGenerator)
                                 (implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver) = {
    val resolvedCi = Try {
      val instanceId = resolveValue(generateInstanceId(template, parentId, idGenerator))
      val ciType = instanceType(template)
      val ci: ConfigurationItem = createCi(ciType, instanceId)

      template.getType.getDescriptor.getPropertyDescriptors.collect {
        case prop if prop.getReferencedType == null && containsPropertyDescriptor(ci, prop) =>
          val value = template.getProperty[AnyRef](prop.getName)
          if (value.nonEmpty) {
            ci.setProperty(prop.getName, resolver.resolve(value, Option(prop.getKind), ignoreContextPlaceholders))
          }
      }
      ci
    }
    alreadyResolvedTemplates.put(template.getId, resolvedCi)
    ResolutionResult(template.getId, Try(resolveValue(generateInstanceId(template, parentId, idGenerator))), resolvedCi)
  }

  private def generateInstanceId(template: Template, parentCiId: ParentCiId, idGenerator: CiIdGenerator): String = {
    if (Option(template.instanceName).exists(_.nonEmpty)) {
      simpleIdGenerator(parentCiId, template.instanceName)
    } else {
      idGenerator(parentCiId, template.getName)
    }
  }

  private def setResolvedReferences(template: Template)(implicit alreadyResolvedTemplates: ResolvedCache, resolver: PlaceholderResolver): Unit = {
    alreadyResolvedTemplates
      .get(template.getId)
      .foreach {
      case Success(resolvedCi) =>
        template.getType.getDescriptor.getPropertyDescriptors
          .filter(
            prop => prop.getReferencedType != null &&
              prop.get(template).nonEmpty &&
              containsPropertyDescriptor(resolvedCi, prop))
          .foreach {
            prop => resolvedCi.setProperty(prop.getName, convertReferenceValue(template, prop))
          }
        template.childTemplates.foreach(setResolvedReferences)
      case _ => //ignore
    }
  }

  @SuppressWarnings(Array("all")) // FIXME: Traversable.head is considered unsafe
  private def convertReferenceValue(template: Template, propertyDescriptor: PropertyDescriptor)(implicit alreadyResolvedTemplates: ResolvedCache): AnyRef = {
    propertyDescriptor.getKind match {
      case PropertyKind.CI =>
        val parentTemplate = propertyDescriptor.get(template).asInstanceOf[Template]
        findAllReferences(Seq(parentTemplate)).head
      case PropertyKind.LIST_OF_CI if isTemplateReference(propertyDescriptor) =>
        new JArrayList(findAllReferences(propertyDescriptor.get(template).asInstanceOf[JList[Template]]))
      case PropertyKind.SET_OF_CI if isTemplateReference(propertyDescriptor) =>
        new JHashSet(findAllReferences(propertyDescriptor.get(template).asInstanceOf[JSet[Template]]))
      case PropertyKind.LIST_OF_CI | PropertyKind.SET_OF_CI =>
        propertyDescriptor.get(template)
    }
  }

  @SuppressWarnings(Array("all")) // FIXME: Option[Try[ConfigurationItem]] is weird type, also Try.get is considered unsafe
  private def findAllReferences(templates: Iterable[Template])(implicit alreadyResolvedTemplates: ResolvedCache): Iterable[ConfigurationItem] =
    templates.map{t =>
      alreadyResolvedTemplates.get(t.getId).map {
        case Success(ci) => ci
        case Failure(e) => throw new IllegalStateException(s"Could not create Configuration Item from template '${t.getId}'. Error: ${e.getMessage}")
      }.get
    }

  private def containsPropertyDescriptor(ci: ConfigurationItem, propertyDescriptor: PropertyDescriptor): Boolean = {
    ci.getType.getDescriptor.getPropertyDescriptor(propertyDescriptor.getName) != null
  }

  private def isTemplateReference(templatePropertyDescriptor: PropertyDescriptor) = {
    templatePropertyDescriptor.getReferencedType.getDescriptor.isAssignableTo(typeOf[Template])
  }

  private def instanceType(template: Template) = {
    typeOf(template.getType.toString.substring(templatePrefix.length))
  }

  def resolveValue(property: String)(implicit resolver: PlaceholderResolver = placeholderResolver()): String = {
    resolver.resolve(property).asInstanceOf[String]
  }
}
