package com.xebialabs.xldeploy.auth

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty
import com.xebialabs.deployit.engine.spi.event.LogoutEvent
import com.xebialabs.deployit.event.EventBusHolder
import org.eclipse.jetty.http.HttpCookie
import org.springframework.security.core.Authentication
import org.springframework.security.core.userdetails.UserDetailsService
import org.springframework.security.web.authentication.rememberme.PersistentTokenBasedRememberMeServices

import jakarta.servlet.http.{Cookie, HttpServletRequest, HttpServletResponse}

class XlPersistentTokenBasedRememberMeServices(key: String,
                                               userDetailsService: UserDetailsService,
                                               tokenRepository: JdbcRememberMeTokenRepository)
  extends PersistentTokenBasedRememberMeServices(key, userDetailsService, tokenRepository) {

  val serverConfiguration: ServerConfiguration = ServerConfiguration.getInstance

  override def logout(request: HttpServletRequest, response: HttpServletResponse, authentication: Authentication): Unit = {
    super.logout(request, response, authentication)
    //publish successful logout event
    if (null != authentication && isNotEmpty(authentication.getName))
      EventBusHolder.publish(new LogoutEvent(authentication.getName))
    // removing tokens because the default spring implementation uses the username from the Authentication
    if (authentication == null) {
      request.getCookies.find(_.getName == this.getCookieName).map { c =>
        val values = decodeCookie(c.getValue)
        values.headOption.map(tokenRepository.removeBySeries)
      }
    }
  }

  override def setCookie(tokens: Array[String], maxAge: Int, request: HttpServletRequest, response: HttpServletResponse): Unit = {
    val cookieValue = encodeCookie(tokens)
    val cookie = new Cookie(this.getCookieName, cookieValue)
    cookie.setMaxAge(maxAge)
    cookie.setPath(getCookiePath(request))
    if (maxAge < 1) {
      cookie.setVersion(1)
    }
    if (!serverConfiguration.isSsl) {
      cookie.setSecure(serverConfiguration.isSecureCookieEnabled)
    } else {
      cookie.setSecure(request.isSecure)
    }
    cookie.setHttpOnly(true)
    cookie.setComment(HttpCookie.SAME_SITE_STRICT_COMMENT)
    response.addCookie(cookie)
  }

  private def getCookiePath(request: HttpServletRequest): String = {
    val contextPath = request.getContextPath
    if (contextPath.nonEmpty) contextPath else "/"
  }

}
