package com.xebialabs.xldeploy.auth.config

import com.xebialabs.deployit.MissingLoginPermissionErrorHandler
import com.xebialabs.deployit.core.auth.LoginMetadataService
import com.xebialabs.deployit.plumbing.authentication.BasicAuthenticationForbiddenEntryPoint
import com.xebialabs.deployit.security.authentication.BasicAuthenticationEntryPoint
import com.xebialabs.deployit.security.{RoleService}
import com.xebialabs.xldeploy.auth.{BasicAuthOverridingHttpSessionSecurityContextRepository, JdbcRememberMeTokenRepositoryImpl, XlPersistentTokenBasedRememberMeServices}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.http.HttpStatus
import org.springframework.security.core.AuthenticationException
import org.springframework.security.core.userdetails.UserDetailsService
import org.springframework.security.web.AuthenticationEntryPoint
import org.springframework.security.web.access.AccessDeniedHandler
import org.springframework.security.web.authentication.logout._
import org.springframework.security.web.authentication.session.SessionAuthenticationException
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, DelegatingAuthenticationEntryPoint, HttpStatusEntryPoint, SimpleUrlAuthenticationFailureHandler}
import org.springframework.security.web.util.matcher.{RequestHeaderRequestMatcher, RequestMatcher}

import java.io.IOException
import java.util.{UUID, LinkedHashMap => JLinkedHashMap}
import jakarta.servlet.ServletException
import jakarta.servlet.http.{HttpServletRequest, HttpServletResponse}

@Configuration
class XLSecurityConfig {

  @Autowired
  var jdbcRememberMeTokenRepositoryImpl: JdbcRememberMeTokenRepositoryImpl = _

  @Autowired
  var userDetailsService: UserDetailsService = _

  @Bean
  def rememberMeKey: String = UUID.randomUUID().toString

  @Bean
  def withoutRedirectLogoutSuccessHandler: HttpStatusReturningLogoutSuccessHandler =
    new HttpStatusReturningLogoutSuccessHandler()

  @Bean
  def delegatingSecurityContextRepository: BasicAuthOverridingHttpSessionSecurityContextRepository =
    new BasicAuthOverridingHttpSessionSecurityContextRepository()

  @Bean
  def rememberMeServices: XlPersistentTokenBasedRememberMeServices = {
    val services = new XlPersistentTokenBasedRememberMeServices(
      rememberMeKey,
      userDetailsService,
      jdbcRememberMeTokenRepositoryImpl)
    services.setTokenValiditySeconds(31556926)
    services.setCookieName("remember-me")
    services
  }

  @Bean
  @Autowired
  def accessDeniedHandler(loginMetadataService: LoginMetadataService, roleService: RoleService): AccessDeniedHandler = {
    new MissingLoginPermissionErrorHandler(loginMetadataService, roleService)
  }

  @Bean
  def delegatingAuthenticationEntryPoint: DelegatingAuthenticationEntryPoint = {
    val entryPoints = new JLinkedHashMap[RequestMatcher, AuthenticationEntryPoint]
    entryPoints.put(
      new RequestHeaderRequestMatcher("X-HTTP-Auth-Override", "true"),
      new BasicAuthenticationForbiddenEntryPoint()
    )
    entryPoints.put(
      new RequestHeaderRequestMatcher("Accept", "application/json"),
      new HttpStatusEntryPoint(HttpStatus.UNAUTHORIZED)
    )

    val point = new DelegatingAuthenticationEntryPoint(entryPoints)
    point.setDefaultEntryPoint(new BasicAuthenticationEntryPoint())
    point
  }

  @Bean
  def authenticationFailureHandler: AuthenticationFailureHandler = {
    new SimpleUrlAuthenticationFailureHandler() {
      @throws[IOException]
      @throws[ServletException]
      override def onAuthenticationFailure(request: HttpServletRequest, response: HttpServletResponse, exception: AuthenticationException): Unit = {
        exception match {
          case exception: SessionAuthenticationException => response.sendError(HttpStatus.UNAUTHORIZED.value, exception.getMessage)
          case _ => super.onAuthenticationFailure(request, response, exception)
        }
      }
    }
  }
}
