package com.xebialabs.xldeploy.auth

import java.sql.ResultSet
import java.util.Date

import com.xebialabs.deployit.core.sql._
import javax.sql.DataSource
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.{DataAccessException, EmptyResultDataAccessException, IncorrectResultSizeDataAccessException}
import org.springframework.jdbc.core.RowMapper
import org.springframework.jdbc.core.support.JdbcDaoSupport
import org.springframework.security.web.authentication.rememberme.{PersistentRememberMeToken, PersistentTokenRepository}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

trait JdbcRememberMeTokenRepository extends PersistentTokenRepository {
  def removeBySeries(seriesId: String): Int
}

@Component
@Transactional(value = "mainTransactionManager")
class JdbcRememberMeTokenRepositoryImpl (@Autowired @Qualifier("mainDataSource") val dataSource: DataSource,
                                         @Autowired @Qualifier("mainSchema") val schemaInfo: SchemaInfo)
  extends JdbcDaoSupport with JdbcRememberMeTokenRepository with PersistentLoginsQueries {
  super.setDataSource(dataSource)

  override def updateToken(series: String, tokenValue: String, lastUsed: Date): Unit =
    getJdbcTemplate.update(UPDATE_TOKEN, tokenValue, lastUsed, series)

  override def getTokenForSeries(seriesId: String): PersistentRememberMeToken = {
    queryBySeries(seriesId, (rs: ResultSet, rowNum: Int) =>
      new PersistentRememberMeToken(rs.getString(1), rs.getString(2), rs.getString(3), rs.getTimestamp(4)), null)
  }

  override def createNewToken(token: PersistentRememberMeToken): Unit =
    getJdbcTemplate.update(CREATE_TOKEN, token.getUsername, token.getSeries, token.getTokenValue, token.getDate)

  override def removeUserTokens(username: String): Unit = getJdbcTemplate.update(DELETE_TOKEN, username)

  override def removeBySeries(seriesId: String): Int  = {
    queryBySeries(seriesId, (rs: ResultSet, rowNum: Int) => getJdbcTemplate.update(DELETE_TOKEN, rs.getString(1)), 0)
  }

  private def queryBySeries[T](seriesId: String, block: RowMapper[T], defaultValue: T): T = {
    try {
      return getJdbcTemplate.queryForObject(TOKEN_BY_SERIES, block, seriesId)
    } catch {
      case zeroResults: EmptyResultDataAccessException =>
        logger.debug(s"Querying token for series '$seriesId' returned no results.", zeroResults)
      case moreThanOne: IncorrectResultSizeDataAccessException =>
        logger.error(s"Querying token for series '$seriesId' returned more than one value. Series should be unique")
      case e: DataAccessException =>
        logger.error(s"Failed to load token for series $seriesId", e)
    }
    defaultValue
  }

}

trait PersistentLoginsQueries extends Queries {

  import PERSISTENT_LOGINS._

  lazy val UPDATE_TOKEN: String = sqlb"update $tableName set $token = ?, $last_used = ? where $series = ?"
  lazy val CREATE_TOKEN: String = sqlb"insert into $tableName ($username, $series, $token, $last_used) values(?,?,?,?)"
  lazy val DELETE_TOKEN: String = sqlb"delete from $tableName where $username = ?"
  lazy val TOKEN_BY_SERIES: String = sqlb"select $username, $series, $token, $last_used from $tableName where $series = ?"

}

object PERSISTENT_LOGINS {
  val tableName = TableName("PERSISTENT_LOGINS", true)

  val token = ColumnName("token", true)
  val last_used = ColumnName("last_used", true)
  val series = ColumnName("series", true)
  val username = ColumnName("username", true)
}
