package com.xebialabs.deployit.core.service.status

import com.xebialabs.deployit.core.events.{ApplicationDeploymentPackageStateEvent, CreateApplicationDeploymentPackage, CreateDeployedApplicationPackage, DeleteApplicationDeploymentPackage, DeleteDeployedApplicationPackage}
import com.xebialabs.deployit.engine.api.dto.{ApplicationDeploymentPackageState, DeploymentPackageState, VersionTag}
import grizzled.slf4j.Logging
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.stereotype.Component

import javax.jms.TextMessage

@Component
class DeploymentPackageStatusConsumer(
                                       jacksonJmsMessageConverter: MessageConverter,
                                       deploymentPackageStatusService: DeploymentPackageStatusService
                                     ) extends Logging {
  @JmsListener(
    destination = "#{@deploymentPackageStatusQueueNameResolver.getDeploymentPackageStatusQueueName()}",
    containerFactory = "xlJmsListenerContainerFactory"
  )
  def receiveTextMessage(message: TextMessage): Unit =
    jacksonJmsMessageConverter.fromMessage(message) match {
      case createEvent: CreateApplicationDeploymentPackage =>
        deploymentPackageStatusService.createApplicationDeploymentPackage(createEvent.applicationDeploymentPackage)
      case deleteEvent: DeleteApplicationDeploymentPackage =>
        deploymentPackageStatusService.deleteApplicationDeploymentPackage(deleteEvent.applicationDeploymentPackage)
      case stateChange: ApplicationDeploymentPackageStateEvent =>
        deploymentPackageStatusService.changeApplicationDeploymentPackageState(
          ApplicationDeploymentPackageState(
            stateChange.applicationName,
            stateChange.applicationPath,
            DeploymentPackageState(
              stateChange.destination,
              "",
              VersionTag(stateChange.versionTag.label, ""),
              stateChange.deploymentStatus.toString,
              stateChange.deploymentType,
              stateChange.user,
              stateChange.lastChangeTime
            )
          )
        )
      case createDeployedApplicationEvent: CreateDeployedApplicationPackage =>
        deploymentPackageStatusService.createDeployedApplicationPackage(createDeployedApplicationEvent.deployedApplicationPackage)
      case deleteDeployedApplicationEvent: DeleteDeployedApplicationPackage =>
        deploymentPackageStatusService.deleteDeployedApplicationPackage(deleteDeployedApplicationEvent.deployedApplicationPackage)
      case _ =>
        logger.error(s"Received unknown ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} [$message]")
    }
}
