package com.xebialabs.deployit.core.listener.version

import com.xebialabs.deployit.engine.api.dto.{ApplicationDeploymentPackage, StateOperation, VersionTag}
import com.xebialabs.deployit.engine.spi.event._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.listener.version.readApplication
import com.xebialabs.deployit.plugin.api.udm.Version
import com.xebialabs.xldeploy.status.queue.DeploymentPackageStatusStoreHolder
import grizzled.slf4j.Logging
import nl.javadude.t2bus.Subscribe

import scala.jdk.CollectionConverters._

@DeployitEventListener
class VersionPropagationListener extends Logging {

  import com.xebialabs.deployit.core.util.IdExtensions._

  @Subscribe
  def receiveCisCreated(event: CisCreatedEvent): Unit = {
    event.getCis.asScala.foreach {
      case version: Version =>
        val application = readApplication(version)
        createApplicationDeploymentPackage(application.getName, application.get$referenceId(), version.getVersion)
      case _ =>
    }
  }

  @Subscribe
  def receiveCisCopied(event: CiCopiedEvent): Unit =
    event.getCi match {
      case version: Version =>
        val application = readApplication(version)
        val oldVersion = event.getEntity.getName
        val newVersion = event.getNewId.getName
        updateApplicationDeploymentPackage(application.getName, application.get$referenceId(), oldVersion, newVersion)
      case _ =>
    }

  @Subscribe
  def receiveCisDeleted(event: CisDeletedEvent): Unit =
    event.getCis.asScala.foreach {
      case version: Version =>
        val application = try {
          readApplication(version)
        } catch {
          case e: NotFoundException => event.getCis.asScala.find(ci => version.getId.startsWith(ci.getId)).getOrElse(throw e)
          case e: Throwable => throw e
        }
        deleteApplicationDeploymentPackage(application.getName, application.get$referenceId(), version.getVersion)
      case _ =>
    }

  @Subscribe
  def receiveCiRenamed(event: CiRenamedEvent): Unit =
    event.getCi match {
      case version: Version =>
        val application = readApplication(version)
        val oldVersion = event.getCi.getName
        val newVersion = event.getNewName
        updateApplicationDeploymentPackage(application.getName, application.get$referenceId(), oldVersion, newVersion)
      case _ =>
    }

  private def updateApplicationDeploymentPackage(applicationName: String, appUid: String, oldVersion: String, newVersion: String): Unit = {
    deleteApplicationDeploymentPackage(applicationName, appUid, oldVersion)
    createApplicationDeploymentPackage(applicationName, appUid, newVersion)
  }

  private def createApplicationDeploymentPackage(applicationName: String, appUid: String, newVersion: String): Unit = {
    DeploymentPackageStatusStoreHolder.getDeploymentPackageStatusStore.sendDeploymentPackageEvent(
      ApplicationDeploymentPackage(StateOperation.CreateState, applicationName, appUid, VersionTag(newVersion, ""))
    )
  }

  private def deleteApplicationDeploymentPackage(applicationName: String, appUid: String, oldVersion: String): Unit =
    DeploymentPackageStatusStoreHolder.getDeploymentPackageStatusStore.sendDeploymentPackageEvent(
      ApplicationDeploymentPackage(StateOperation.RemoveState, applicationName, appUid, VersionTag(oldVersion, ""))
    )
}
