package com.xebialabs.deployit.core.rest.json;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.validation.PatternValidator;
import com.xebialabs.deployit.plugin.api.validation.Range;
import com.xebialabs.deployit.plugin.api.validation.Validator;
import com.xebialabs.xltype.serialization.json.JsonWriter;
import org.springframework.stereotype.Component;

import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.ext.MessageBodyWriter;
import javax.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

@Component
@Provider
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public class PropertyDescriptorJsonWriter implements MessageBodyWriter<PropertyDescriptor> {

    private static void writeValue(JsonWriter writer, String key, Object value) {
        if (value instanceof Collection) {
            writer.key(key);
            writer.array();
            for (Object current : (Iterable) value) {
                writer.value(current);
            }
            writer.endArray();
        } else if (value instanceof Map) {
            writer.key(key);
            writer.object();
            for (Map.Entry<String, String> entry : ((Map<String, String>) value).entrySet()) {
                writer.key(entry.getKey());
                writer.value(entry.getValue());
            }
            writer.endObject();
        } else {
            writer.key(key).value(value);
        }
    }

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return PropertyDescriptor.class.isAssignableFrom(type);
    }

    @Override
    public long getSize(PropertyDescriptor propertyDescriptor, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return -1L;
    }

    @Override
    public void writeTo(PropertyDescriptor propertyDescriptor, Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders, OutputStream entityStream) throws IOException, WebApplicationException {
        entityStream.write(toJson(propertyDescriptor).getBytes());
    }

    @VisibleForTesting
    String toJson(PropertyDescriptor propertyDescriptor) {
        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writePropertyDescriptor(writer, propertyDescriptor);
        return stringWriter.toString();
    }

    static void writePropertyDescriptor(JsonWriter writer, PropertyDescriptor propertyDescriptor) {
        writer.object();
        writer.key("name").value(propertyDescriptor.getName());
        writer.key("fqn").value(propertyDescriptor.getFqn());
        writer.key("label").value(propertyDescriptor.getLabel());
        writer.key("kind").value(propertyDescriptor.getKind().name());
        writer.key("description").value(propertyDescriptor.getDescription());
        writer.key("category").value(propertyDescriptor.getCategory());
        writer.key("asContainment").value(propertyDescriptor.isAsContainment());
        writer.key("inspection").value(propertyDescriptor.isInspectionProperty());
        writer.key("required").value(propertyDescriptor.isRequired());
        writer.key("requiredInspection").value(propertyDescriptor.isRequiredForInspection());
        writer.key("password").value(propertyDescriptor.isPassword());
        writer.key("transient").value(propertyDescriptor.isTransient());
        writer.key("size").value(propertyDescriptor.getSize().name());
        writer.key("referencedType").value(propertyDescriptor.getReferencedType());
        writeValue(writer, "default", propertyDescriptor.getDefaultValue());
        if (propertyDescriptor.isHidden()) {
            writer.key("hidden").value(true);
        }
        if (propertyDescriptor.isReadonly()) {
            writer.key("readonly").value(true);
        }
        writeValue(writer, "enumValues", propertyDescriptor.getEnumValues());
        writeInputHint(writer, propertyDescriptor);
        writer.endObject();
    }

    private static void writeInputHint(final JsonWriter writer, final PropertyDescriptor propertyDescriptor) {
        if (propertyDescriptor.getInputHint() != null) {
            writer.key("inputHint");
            writer.object();
            writer.key("kind").value(propertyDescriptor.getInputHint().getKind());
            if (propertyDescriptor.getInputHint().getValues() != null) {
                writer.key("values");
                writer.array();
                propertyDescriptor.getInputHint().getValues().forEach(inputHintValue -> {
                    writer.object();
                    writer.key("value").value(inputHintValue.getValue());
                    writer.key("label").value(inputHintValue.getLabel());
                    writer.endObject();
                });
                writer.endArray();
            }
            writer.key("rules");
            writer.array();
            Set<Validator<?>> validationRules = propertyDescriptor.getInputHint().getValidationRules();
            if (validationRules != null && !validationRules.isEmpty()) {
                validationRules
                        .stream()
                        .filter(v -> v instanceof PatternValidator || v instanceof Range.Validator)
                        .forEach(patternValidator -> writeInputHintRules(writer, patternValidator));
            }
            writer.endArray();
            writer.key("prompt").value(propertyDescriptor.getInputHint().getPrompt());
            writer.key("required").value(propertyDescriptor.getInputHint().isRequired());
            writer.key("copy-from-property").value(propertyDescriptor.getInputHint().getCopyFromProperty());
            writer.endObject();
        }
    }

    private static void writeInputHintRules(final JsonWriter writer, final Validator validator) {
        if (validator instanceof PatternValidator) {
            writeInputHintPatternRule(writer, (PatternValidator) validator);
        } else if (validator instanceof Range.Validator) {
            writeInputHintRangeRule(writer, (Range.Validator) validator);
        }
    }

    private static void writeInputHintRangeRule(final JsonWriter writer, final Range.Validator validator) {
        writer.object();
        writer.key("type").value("range");
        writer.key("message").value(validator.getMessage());
        writer.key("min").value(validator.getMinimum());
        writer.key("max").value(validator.getMaximum());
        writer.endObject();
    }

    private static void writeInputHintPatternRule(final JsonWriter writer, final PatternValidator validator) {
        writer.object();
        writer.key("type").value("pattern");
        writer.key("message").value(validator.getMessage());
        writer.key("pattern").value(validator.getPattern());
        writer.endObject();
    }
}
