package com.xebialabs.deployit.core.service.impl

import ai.digital.deploy.task.status.util.DeploymentStatusUtils
import ai.digital.deploy.tasker.common.{TaskMetadata, TaskType}
import com.xebialabs.deployit.core.events.dto.{ApplicationDeploymentPackageState, DeploymentPackageState}
import com.xebialabs.deployit.core.rest.api.RepositoryResource
import com.xebialabs.deployit.core.service.ApplicationStatusService
import com.xebialabs.deployit.engine.api.dto.Ordering
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskExecutionState, TaskWithBlock}
import com.xebialabs.deployit.engine.tasker.TaskExecutionEngine
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.repository.sql.CiRepository
import com.xebialabs.deployit.repository.sql.base.idToPath
import com.xebialabs.deployit.repository.{DeployedApplicationsRepository, SearchParameters, WorkDir}
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.security.client.PermissionService
import com.xebialabs.deployit.security.permission.{PermissionHelper, PlatformPermissions}
import com.xebialabs.deployit.spring.BeanWrapper
import com.xebialabs.overthere.local.LocalFile
import com.xebialabs.xlplatform.sugar.TempDirectorySugar
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{UUID, List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Service
class ApplicationStatusServiceImpl @Autowired()(deployedApplicationsRepository: DeployedApplicationsRepository,
                                                ciRepository: CiRepository,
                                                engine: BeanWrapper[TaskExecutionEngine],
                                                permissionService: PermissionService,
                                                repositoryResource: RepositoryResource)

  extends ApplicationStatusService with TempDirectorySugar with Logging {

  val defaultOrderingField = "applicationName"
  val defaultOrderingDirection = "ASC"

  val internalRoot = "Applications"

  override def getApplicationStatus(deployedAppName: String, path: String, exactPath: Boolean = false, order: Ordering, folders: List[String], appRegex: String, envRegex: String): JList[ApplicationDeploymentPackageState] = {
    val (orderingField, direction) = if (order == null) (defaultOrderingField, defaultOrderingDirection) else (order.field, order.direction.toString)

    val deployedAppStates = getDeployedApplicationStates(deployedAppName, path, exactPath, folders, appRegex, envRegex)
    val taskStates = getTaskStates(deployedAppName, path, exactPath, folders, appRegex, envRegex)

    val applicationStates = squashResult(List(deployedAppStates, taskStates), orderingField)
    val applicationStatesData = direction match {
      case "DESC" => applicationStates.reverse.asJava
      case _ => applicationStates.asJava
    }

    applicationStatesData
  }

  override def getFilterableFolders: JList[String] = {
    val criteria = new SearchParameters()
      .setType(Type.valueOf(classOf[Directory]))
      .setAncestor(internalRoot)
    List.concat(List(internalRoot), ciRepository.listEntities[Directory](criteria).asScala.map(_.getId)).asJava
  }


  private def getDeployedApplicationStates(deployedAppName: String, path: String, exactPath: Boolean, folders: List[String], appRegex: String, envRegex: String): Map[(String, String), Map[(String, String), DeploymentDetails]] = {
    val paths = deployedApplicationsRepository.find(deployedAppName, path, exactPath).map(_.getId).asJava
    val deployedApplications = withTempDirectory { tempDir =>
      val workDir = new WorkDir(LocalFile.from(tempDir.toFile))
      ciRepository.read[DeployedApplication](paths, workDir, 2,
        useCache = true,
        decryptPasswords = false,
        skipNotExistingCis = true).asScala
    }

    val filteredDeployedApplications = deployedApplications.filter(deployedApp => {
        filterByFolderFilter(idToPath(DeploymentStatusUtils.getApplicationDirectoryFromVersionId(deployedApp.getVersion.getId)), folders) &&
        filterByAppRegexFilter(deployedApp.getVersion.getApplication.getId, appRegex) &&
        filterByEnvRegexFilter(deployedApp.getEnvironment.getId, envRegex)
    })

    mapDeployedApplicationsToDeploymentDetails(filteredDeployedApplications)
  }

  private def squashResult(data: List[Map[(String, String), Map[(String, String), DeploymentDetails]]], ordering: String): List[ApplicationDeploymentPackageState] =
    data.flatten.groupMapReduce(_._1)(_._2)(reduceEnvs)
      .flatMap {
        case ((appName, appUid), details) =>
          details.map {
            case ((envName, envUid), detail) =>
              ApplicationDeploymentPackageState(
                appName,
                appUid,
                detail.applicationPath,
                DeploymentPackageState(
                  envName,
                  envUid,
                  "",
                  detail.version,
                  detail.state,
                  detail.deploymentType,
                  detail.user,
                  detail.time)
              )
          }
      }.toList.sortBy(s => ordering match {
      case "applicationName" => s.applicationName
      case "destination" => s.state.destination
      case "versionTag" => s.state.versionTag
      case "deploymentStatus" => s.state.deploymentStatus
      case "user" => s.state.user
      case "lastChangeTime" => s.state.lastChangeTime.getMillis.toString
      case _ => s.applicationName
    })

  private def reduceEnvs: (Map[(String, String), DeploymentDetails], Map[(String, String), DeploymentDetails]) => Map[(String, String), DeploymentDetails] =
    (done, ongoing) => {
      List(done, ongoing).flatten.groupMapReduce(_._1)(_._2)((_, o) => o)
    }


  private def mapDeployedApplicationsToDeploymentDetails(deployedApplications: mutable.Buffer[DeployedApplication]):  Map[(String, String), Map[(String, String), DeploymentDetails]]  = {
    import com.xebialabs.deployit.core.util.IdExtensions._

    deployedApplications.map(app => {
      val application = repositoryResource.read(app.getVersion.getId.getParent)
      (app.getName, application.get$referenceId()) ->
        ((app.getEnvironment.getName, app.getEnvironment.get$referenceId()) ->
          DeploymentDetails(
            app.getVersion.getVersion,
            TaskExecutionState.DONE.toString,
            "",
            app.get$ciAttributes().getLastModifiedBy,
            app.get$ciAttributes().getLastModifiedAt,
            DeploymentStatusUtils.getApplicationPathFromVersionId(app.getVersion.getId)
          )
          )
    }
    ).groupBy(_._1).map {
      case (appName, values) =>
        appName -> values.map(_._2).toMap
    }
  }

  private def filterByDeploymentAppName(task: TaskWithBlock, deployedAppName: String): Boolean =
    Option(deployedAppName) match {
      case Some(value) => DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION)(task).contains(value)
      case None => true
    }

  private def filterByPathFilter(task: TaskWithBlock, path: String, exactPath: Boolean): Boolean =
    Option(path) match {
      case Some(value) =>
        try {
          val applicationPath = idToPath(DeploymentStatusUtils.getApplicationIdFromVersionId(DeploymentStatusUtils.extractMetadata(TaskMetadata.VERSION_ID)(task)))
          val filterPath = idToPath(value)
          if (exactPath)
            applicationPath.equals(filterPath)
          else
            applicationPath.startsWith(filterPath)
        } catch {
          case e: Exception =>
            logger.error("Exception occurred while filtering by path ", e)
            false
        }
      case None => true
    }

  private def filterByPermissions(task: TaskWithBlock): Boolean = {
    val envDirRef = DeploymentStatusUtils.extractMetadata(TaskMetadata.ENVIRONMENT_DIRECTORY_REFERENCE)(task)
    val appDirRef = DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION_DIRECTORY_REFERENCE)(task)
    hasPermission(List(envDirRef, appDirRef))
  }

  private def getTaskStates(deployedAppName: String, path: String, exactPath: Boolean, folders: List[String], appRegex: String, envRegex: String): Map[(String, String), Map[(String, String), DeploymentDetails]] = {
    val allTasks = engine.get().getAllIncompleteTasks(FetchMode.SUMMARY).asScala.toVector
    allTasks.filter(task =>
      Option(task.getState).isDefined &&
        TaskType.valueOf(DeploymentStatusUtils.extractMetadata(TaskMetadata.TASK_TYPE)(task)) != TaskType.CONTROL &&
        TaskType.valueOf(DeploymentStatusUtils.extractMetadata(TaskMetadata.TASK_TYPE)(task)) != TaskType.DEFAULT &&
        TaskType.valueOf(DeploymentStatusUtils.extractMetadata(TaskMetadata.TASK_TYPE)(task)) != TaskType.INSPECTION &&
        filterByDeploymentAppName(task, deployedAppName) &&
        filterByPathFilter(task, path, exactPath) &&
        filterByFolderFilter(idToPath(DeploymentStatusUtils.getApplicationDirectoryFromVersionId(DeploymentStatusUtils.extractMetadata(TaskMetadata.VERSION_ID)(task))), folders) &&
        filterByAppRegexFilter(DeploymentStatusUtils.getApplicationIdFromVersionId(DeploymentStatusUtils.extractMetadata(TaskMetadata.VERSION_ID)(task)), appRegex) &&
        filterByEnvRegexFilter(DeploymentStatusUtils.extractMetadata(TaskMetadata.ENVIRONMENT_ID)(task), envRegex) &&
        filterByPermissions(task)
    ).map(task =>
          (DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION)(task),
            DeploymentStatusUtils.extractMetadata(TaskMetadata.APPLICATION_REFERENCE_ID)(task))->
          ((DeploymentStatusUtils.extractMetadata(TaskMetadata.ENVIRONMENT)(task),
            DeploymentStatusUtils.extractMetadata(TaskMetadata.ENVIRONMENT_REFERENCE_ID)(task)) ->
            DeploymentDetails(
              DeploymentStatusUtils.extractMetadata(TaskMetadata.VERSION)(task),
              task.getState.toString,
              DeploymentStatusUtils.extractMetadata(TaskMetadata.TASK_TYPE)(task),
              task.getOwner,
              Option(task.getStartDate).orElse(Option(task.getScheduledDate)).getOrElse(
                DateTime.now()
              ),
              DeploymentStatusUtils.getApplicationPathFromTaskId(task.getId)
            )
            )
    ).groupBy(_._1).map {
      case (appKey, values) =>
        appKey -> values.map(_._2).toMap
    }
  }

  private def hasPermission(securedDirectoryReferences: List[String]): Boolean = {
    PermissionHelper.isCurrentUserAdmin || permissionService.checkPermission(
      securedDirectoryReferences.map(UUID.fromString).asJava, List(PlatformPermissions.READ).asJava, Permissions.getAuthentication)
  }

  private def filterByFolderFilter(folderPath: String, folders: List[String]): Boolean =
    Option(folders) match {
      case Some(value) if value.nonEmpty => folders.exists(folder => folderPath.contains(folder))
      case _ => true
    }

  private def filterByAppRegexFilter(applicationId: String, appRegex: String): Boolean =
    Option(appRegex) match {
      case Some(value) => applicationId.matches(value)
      case _ => true
    }

  private def filterByEnvRegexFilter(environmentId: String, envRegex: String): Boolean =
    Option(envRegex) match {
      case Some(value) => environmentId.matches(value)
      case _ => true
    }
}

final case class DeploymentDetails(version: String, state: String, deploymentType: String, user: String, time: DateTime, applicationPath: String)
