package com.xebialabs.deployit.core.rest.websockets

import com.xebialabs.deployit.core.events.{CiRef, JmsCiChangeSetEvent}
import com.xebialabs.deployit.core.websockets.WebSocketEndpoints
import com.xebialabs.deployit.engine.spi.event.{MaintenanceStartEvent, MaintenanceStopEvent}
import com.xebialabs.deployit.event.EventBusHolder
import nl.javadude.t2bus.Subscribe
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.messaging.simp.SimpMessagingTemplate
import org.springframework.stereotype.Controller

import java.util.{Collection => JavaCollection}
import scala.jdk.CollectionConverters._

@Autowired
@Controller
class WebSocketMessagesSender(@Autowired messageTemplate: SimpMessagingTemplate) {
  EventBusHolder.register(this)

  @Subscribe
  def sendWsCiChangeSetMessages(event: JmsCiChangeSetEvent): Unit = {
    val username = event.username
    notifyCisIfNotEmpty(WebSocketEndpoints.TOPIC_CI_CREATE, event.createCis.asJava, username)
    notifyCisIfNotEmpty(WebSocketEndpoints.TOPIC_CI_UPDATE, event.updateCis.asJava, username)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_DELETE, event.deleteCis.asJava, username)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_RENAME, event.renameCis.asJava, username)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_MOVE, event.moveCis.asJava, username)
    notifyWithItemsIfNotEmpty(WebSocketEndpoints.TOPIC_CI_COPY, event.copyCis.asJava, username)
  }

  @Subscribe
  def sendMaintenanceStarted(event: MaintenanceStartEvent): Unit = {
    notifyAboutMaintenanceModeChange(true)
  }

  @Subscribe
  def sendMaintenanceStopped(event: MaintenanceStopEvent): Unit = {
    notifyAboutMaintenanceModeChange(false)
  }

  private def notifyAboutMaintenanceModeChange(enabled: Boolean): Unit = {
    messageTemplate.convertAndSend(WebSocketEndpoints.TOPIC_MAINTENANCE.getName, enabled)
  }

  private def notifyCisIfNotEmpty(endpoint: WebSocketEndpoints, ciIds: JavaCollection[CiRef], username: String): Unit = {
    notifyWithItemsIfNotEmpty(endpoint, ciIds, username)
  }

  private def notifyWithItemsIfNotEmpty[T](endpoint: WebSocketEndpoints, items: JavaCollection[T], username: String): Unit = {
    if (!items.isEmpty) {
      messageTemplate.convertAndSendToUser(username, endpoint.getName, items)
    }
  }

}
