package com.xebialabs.deployit.core.rest.api

import com.xebialabs.deployit.core.api.XldUserTokenService
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.security.model.{Tokens, XldUserToken}
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.deployit.security.service.UserTokenService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Controller
import ai.digital.deploy.sql.model.{CreatePersonalAccessTokenForm, UserTokens, UserTokensView}
import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.UserService

import scala.jdk.CollectionConverters._

object XldUserTokenService {
  def mapToUserTokenView(tokens: Tokens): UserTokens = {
    new UserTokens(
      tokens.createdDate,
      tokens.lastUsedDate,
      tokens.expiryDate,
      tokens.tokenNote,
      tokens.ciId
    )
  }
  def mapToUserTokenViewForCreateToken(tokens: Tokens): UserTokens = {
    new UserTokens(
      tokens.createdDate,
      tokens.lastUsedDate,
      tokens.expiryDate,
      tokens.tokenNote,
      tokens.tokenHash,
      tokens.ciId
    )
  }
}
@Controller
class XldUserTokenServiceImpl @Autowired()(userTokenService: UserTokenService, userService: UserService) extends  AbstractUserRestrictedResource with XldUserTokenService {

  override def getUserTokens(paging: Paging, order: Ordering, tokenNote: String): UserTokensView = {
    val authentication = SecurityContextHolder.getContext.getAuthentication
    val userTokens = userTokenService.listUserTokenByUserName(authentication.getName, paging, order, tokenNote)
    if (userTokens == null) {
      return new UserTokensView(authentication.getName, List.empty.asJava)
    }
    new UserTokensView(
      userTokens.username,
      userTokens.tokens.map(tokens => XldUserTokenService.mapToUserTokenView(tokens)).asJava,
    )
  }

  override def getTokensForUsername(username: String, paging: Paging, order: Ordering, tokenNote: String): UserTokensView = {
    checkCurrentUserOrPermissions(username, ADMIN)
    val userTokens: XldUserToken = userTokenService.listUserTokenByUserName(username, paging, order, tokenNote)
    if (userTokens == null) {
      return new UserTokensView(username, List.empty.asJava)
    }
    new UserTokensView(
      userTokens.username,
      userTokens.tokens.map(tokens => XldUserTokenService.mapToUserTokenView(tokens)).asJava,
    )
  }

  override def generateUserToken(userTokenForm: CreatePersonalAccessTokenForm): UserTokensView = {

    val username = SecurityContextHolder.getContext.getAuthentication.getName
    try{
      userService.read(username)
    } catch {
      case e: NotFoundException =>
        throw new NotFoundException("Token generation is disabled for LDAP/OIDC user")
    }

    checkArgument(userTokenForm.tokenNote.nonEmpty, "Note can't be blank")
    checkArgument(userTokenForm.expiryDate == null || userTokenForm.expiryDate.getTime > System.currentTimeMillis(), "Expiry date is not valid")

    val tokenNote = userTokenForm.tokenNote.trim
    userTokenForm.tokenNote = tokenNote

    checkArgument(tokenNote.length < 256, "Note must be 255 characters or less")

    val xldUserToken: XldUserToken = userTokenService.listUserTokenByUserName(username, null, null, null)
    checkArgument(!xldUserToken.tokens.exists(token => token.tokenNote == userTokenForm.tokenNote), "Note has already been taken")

    val userTokens: XldUserToken = userTokenService.createUserToken(username, userTokenForm.tokenNote, userTokenForm.expiryDate)
    new UserTokensView(
      userTokens.username,
      userTokens.tokens.map(tokens => XldUserTokenService.mapToUserTokenViewForCreateToken(tokens)).asJava
    )
  }

  override def deleteUserTokenByUsernameAndCiId(username: String, ciId: Integer): Unit = {
    checkCurrentUserOrPermissions(username, ADMIN)
    userTokenService.deleteUserToken(username, ciId)
  }

  override def deleteUserTokenByCiId(ciId: Integer): Unit = {
    val username = SecurityContextHolder.getContext.getAuthentication.getName

    val xldUserToken: XldUserToken = userTokenService.listUserTokenByUserName(username, null, null, null)
    if(xldUserToken.tokens.exists(token => token.ciId == ciId))
      userTokenService.deleteUserToken(username, ciId)
    else
      throw new IncorrectArgumentException(s"Personal access token ciId[$ciId] does not belong to user[$username]")
  }
}
