package com.xebialabs.deployit.core.rest.api.reports.widgets;

import ai.digital.deploy.sql.model.Report;
import ai.digital.deploy.sql.model.ReportLine;
import com.xebialabs.deployit.core.rest.api.reports.ReportUtils;
import com.xebialabs.deployit.task.archive.DeploymentResult;
import com.xebialabs.deployit.task.archive.TaskArchive;
import com.xebialabs.deployit.task.archive.ThroughputReportLine;
import org.joda.time.DateTime;

import java.util.Collection;
import java.util.Map;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.summingInt;

/*********************
 * Throughput widget
 */
public class DeploymentTrendsWidget extends DashboardWidgetBase {
    private static final String PERCENTAGE_SUCCESSFUL_DEPLOYMENTS = "percentageSuccessfulDeployments";

    public DeploymentTrendsWidget(final TaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public final Report getReport(DateTime begin, DateTime end) {
        Collection<ThroughputReportLine> searchResult = taskArchive.throughput(begin, end);
        return transformResult(searchResult, begin, end);
    }

    private Report transformResult(Collection<ThroughputReportLine> searchResult, DateTime begin, DateTime end) {
        Report report = new Report();
        if (end.isBefore(begin)) {
            return report;
        }

        final Map<Integer, Map<Integer, Map<String, Integer>>> searchResultGroupedByYearMonthDeployResult =
                searchResult.stream().collect(groupingBy(ThroughputReportLine::year,
                        groupingBy(ThroughputReportLine::monthno,
                                groupingBy(ThroughputReportLine::executionState, summingInt(ThroughputReportLine::amount)))));

        ReportUtils.getMonthsOfInterval(begin, end).forEach(month -> {
            final Map<Integer, Map<String, Integer>> monthDeployResultCount = searchResultGroupedByYearMonthDeployResult.get(month.getYear());
            int currentMonth = month.monthOfYear().get();

            if (monthDeployResultCount == null) {
                addReportLine(report, currentMonth);
            } else {
                final Map<String, Integer> monthAmounts = monthDeployResultCount.get(currentMonth);

                if (monthAmounts == null) {
                    addReportLine(report, currentMonth);
                } else {
                    addReportLine(report, currentMonth,
                            monthAmounts.getOrDefault(DeploymentResult.ROLLBACK.name(), 0),
                            monthAmounts.getOrDefault(DeploymentResult.SUCCESS.name(), 0),
                            monthAmounts.getOrDefault(DeploymentResult.ABORTED.name(), 0),
                            monthAmounts.getOrDefault(DeploymentResult.FAILED.name(), 0),
                            monthAmounts.getOrDefault(DeploymentResult.TOTAL.name(), 0));
                }
            }
        });
        return report;
    }

    private static void addReportLine(Report report, Integer month) {
        addReportLine(report, month, 0, 0, 0, 0, 0);
    }

    private static void addReportLine(Report report, Integer month,
                                      int rollbackDeployments, int successDeployments,
                                      int abortedDeployments, int failedDeployments, int totalDeployments) {

        ReportLine line = report.addLine();

        line.addValue(MONTH, getFormattedMonth(month));
        line.addValue(NUMBER_ROLLBACK_DEPLOYMENTS, String.valueOf(rollbackDeployments));
        line.addValue(NUMBER_SUCCESSFUL_DEPLOYMENTS, String.valueOf(successDeployments));
        line.addValue(NUMBER_ABORTED_DEPLOYMENTS, String.valueOf(abortedDeployments));
        line.addValue(NUMBER_FAILED_DEPLOYMENTS, String.valueOf(failedDeployments));
        line.addValue(PERCENTAGE_SUCCESSFUL_DEPLOYMENTS, getFormattedPercentage(successDeployments, totalDeployments));
    }

}
