package com.xebialabs.deployit.core.rest.api.reports;

import ai.digital.deploy.sql.model.Report;
import ai.digital.deploy.sql.model.ReportLine;
import ai.digital.deploy.tasker.common.TaskType;
import com.xebialabs.deployit.task.archive.TaskArchive;
import com.xebialabs.xltype.serialization.util.DateUtil;
import org.joda.time.DateTime;

import java.util.*;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks.*;

public class DeploymentsForEnvironmentReport {

    private final TaskArchive taskArchive;

    public DeploymentsForEnvironmentReport(final TaskArchive taskArchive) {
        this.taskArchive = taskArchive;
    }

    public Report report(final String environment, final DateTime before) {
        Stream<Map<String, Object>> tasks = taskArchive.lastDeploymentsOnEnvironment(environment, before, 2);
        Map<String, List<Map<String, Object>>> tasksByApplication = tasks.collect(Collectors.groupingBy(map -> (String) map.get(main_application().name()), toSortedList((lhs, rhs) -> {
            DateTime lhsDate = (DateTime) lhs.get(end_date().name());
            DateTime rhsDate = (DateTime) rhs.get(end_date().name());
            return lhsDate.compareTo(rhsDate);
        })));
        return Report.fromLines(tasksByApplication.values().stream().map(this::fillDeployedAppsInReport).filter(Objects::nonNull));
    }

    private ReportLine fillDeployedAppsInReport(List<Map<String, Object>> taskList) {
        Map<String, Object> deployedApp = null;
        Map<String, Object> latestTask = taskList.get(taskList.size() - 1);
        String task_type = (String) latestTask.get(task_type().name());
        String version = null;
        if (task_type.equals(TaskType.INITIAL.name()) || task_type.equals(TaskType.UPGRADE.name())) {
            deployedApp = latestTask;
            version = version(deployedApp);
        }
        if (task_type.equals(TaskType.ROLLBACK.name()) && taskList.get(0).get(task_type().name()).equals(TaskType.UPGRADE.name())) {
            deployedApp = taskList.get(0);
            version = version(latestTask);
        }
        if (deployedApp != null) {
            ReportLine line = new ReportLine();
            line.addValue("application", deployedApp.get(main_application().name()));
            line.addValue("taskId", deployedApp.get(task_id().name()));
            line.addValue("version", version);
            line.addValue("user", deployedApp.get(owner().name()));
            line.addValue("date", DateUtil.toString((DateTime) deployedApp.get(end_date().name())));
            return line;
        } else {
            return null;
        }
    }

    private String version(Map<String, Object> deployedApp) {
        return ((String) deployedApp.get(packages().name())).split("/")[1];
    }

    private static <T> Collector<T, ?, List<T>> toSortedList(Comparator<? super T> c) {
        return Collectors.collectingAndThen(
                Collectors.toCollection(ArrayList::new), l -> {
                    l.sort(c);
                    return l;
                });
    }
}
