package com.xebialabs.deployit.core.pat

import ai.digital.deploy.notification.XldNotificationStore
import ai.digital.deploy.notification.events.UserTokenAboutToExpireEvent
import ai.digital.deploy.sql.model.UserTokens
import com.xebialabs.deployit.core.service.GeneralSettingsService
import com.xebialabs.deployit.engine.api.dto.Paging
import com.xebialabs.deployit.security.model.{Tokens, XldUserToken}
import com.xebialabs.deployit.security.service.UserTokenService
import org.slf4j.{Logger, LoggerFactory}
import org.springframework.stereotype.Component
import org.springframework.beans.factory.annotation.Autowired
import scala.jdk.CollectionConverters._
import java.time.Instant

@Component
class PatNotificationService (@Autowired val generalSettingsService: GeneralSettingsService,
                              @Autowired val xldNotificationStore: XldNotificationStore,
                              @Autowired val userTokenService: UserTokenService) {

  val logger: Logger = LoggerFactory.getLogger(classOf[PatNotificationService])

  def handlePatNotificationForExpiredToken(): Unit = {
    val startTime = Instant.now()
    logger.info(s"PAT :: Starting PAT notification job at $startTime")

    val generalSettings = generalSettingsService.getGeneralSettings
    if (generalSettings != null && generalSettings.patEmailNotificationEnabled) {
      var page = 1
      var totalProcessed = 0
      val pageSize = 100
      var hasMorePages = true

      while (hasMorePages) {
        val tokens = getTokensExpiringInDays(generalSettings.patEmailNotificationTriggerPeriod.toInt, new Paging(page, pageSize))

        if (tokens == null || tokens.size() == 0) {
          // No more tokens, stop processing
          hasMorePages = false
        } else {
          // Process all tokens in current page
          tokens.asScala
            .flatMap(userToken => userToken.tokens.map(token => (userToken.username, token)))
            .foreach { case (username, token) =>
              xldNotificationStore.handlePatNotificationEvent(
                UserTokenAboutToExpireEvent(username, convertUserTokenToToken(token))
              )
              updateTokenExpiredNotified(token.ciId)
              totalProcessed += 1
            }
          // Check if this was the last page (fewer results than page size)
          if (tokens.size() < pageSize) {
            hasMorePages = false
          } else {
            // Move to next page
            page += 1
          }
        }
      }

      logger.info(s"PAT :: Event triggered to send email notifications for $totalProcessed expiring tokens")
      val endTime = Instant.now()
      val durationMillis = java.time.Duration.between(startTime, endTime).toMillis

      logger.info(s"PAT :: PAT notification job completed at $endTime. Duration: ${durationMillis}ms")
    } else {
      logger.info("PAT :: PAT notification is not enabled in configuration, skipping notification check")
    }
  }

  private[pat] def getTokensExpiringInDays(notifiedDays: Int, paging: Paging): java.util.List[XldUserToken] = {
    userTokenService.listTokensExpiringInDays(notifiedDays, paging)
  }

  private def updateTokenExpiredNotified(cid: Integer): Unit = {
    userTokenService.updateTokenExpiredNotified(cid, java.sql.Timestamp.from(Instant.now()), isNotified = true)
  }

  private def convertUserTokenToToken(tokens: Tokens) : UserTokens  = {
    new UserTokens(
      tokens.createdDate,
      tokens.lastUsedDate,
      tokens.expiryDate,
      tokens.tokenNote,
      tokens.tokenHash,
      tokens.ciId
    )
  }
}
