package com.xebialabs.deployit.core.rest.api.reports;

import ai.digital.deploy.sql.model.Report;
import ai.digital.deploy.sql.model.ReportLine;
import com.xebialabs.deployit.ReleaseInfo;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.rest.api.DownloadResource;
import com.xebialabs.deployit.core.rest.api.reports.mapper.ReportToCsvMapper;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.report.audit.AuditPermissionRoleRow;
import com.xebialabs.deployit.report.audit.RolePrincipalPermissionRow;
import com.xebialabs.deployit.task.archive.StatusOverviewDataReportLine;
import com.xebialabs.deployit.task.archive.StatusOverviewReportLine;
import org.apache.poi.ss.usermodel.Workbook;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.Charset;
import java.util.*;
import java.util.stream.Stream;

import static java.lang.String.format;

public class ReportUtils {
    private static final Logger log = LoggerFactory.getLogger(ReportUtils.class);

    public static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyyMMdd");
    static final PeriodFormatter HH_MM_SS = new PeriodFormatterBuilder().minimumPrintedDigits(2).printZeroAlways().appendHours().appendSeparator(":").appendMinutes().appendSeparator(":").appendSeconds().toFormatter();

    private ReportUtils() {
    }

    static String formatToHoursMinsSecs(long millis) {
        Period period = new Duration(millis).toPeriod();
        return HH_MM_SS.print(period);
    }

    public static Stream<String> toCsv(Set<String> fields, Map<String,String> headersMap, Stream<ReportLine> report) {
        return new ReportToCsvMapper(report, fields,headersMap).getCsv();
    }


    public static List<DateTime> getMonthsOfInterval(DateTime begin, DateTime end) {
        List<DateTime> result = new ArrayList<>();
        DateTime current = begin;
        while (current.isBefore(end) || current.isEqual(begin)) {
            result.add(current);
            current = current.plusMonths(1);
        }
        if (result.size() == 1 && begin.getMonthOfYear() != end.getMonthOfYear()) {
            result.add(end);
        }
        return result;
    }

    public static String createDownloadToken(DownloadResource downloadResource, Stream<ReportLine> reportLines, String fileName) {
        return createDownloadToken(downloadResource, null, null,reportLines, fileName);
    }

    public static String createDownloadToken(DownloadResource downloadResource, Set<String> firstRowFields,Map<String,String> headersMap, Stream<ReportLine> reportLines, String fileName) {
        try {
            File to = new File(WorkdirHolder.get().getPath(), fileName);
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(to), Charset.defaultCharset()));
                 Stream<String> s = toCsv(firstRowFields,headersMap, reportLines)
            ) {
                s.forEachOrdered(line -> {
                    try {
                        writer.write(line);
                        writer.write('\n');
                    } catch (IOException io) {
                        throw new RuntimeException(io);
                    }
                });
            }
            return downloadResource.register(to, "text/csv");
        } catch (Exception re) {
            log.error("Oops!", re);
            WorkdirHolder.get().delete();
            throw new IllegalStateException(re);
        }
    }

    public static String createDownloadToken(DownloadResource downloadResource, Report report, String fileName) {
        return createDownloadToken(downloadResource, report.getLines().stream(), fileName);
    }

    public static String createStepLogs(String fileName, DownloadResource downloadResource, List<String> previousLogs, String currentLog) {
        File to = new File(WorkdirHolder.get().getPath(), fileName);
        try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(to), Charset.defaultCharset()))) {
            if (previousLogs != null && !previousLogs.isEmpty()) {
                for (int i = 0; i < previousLogs.size(); i++) {
                    writer.write("#");
                    writer.newLine();
                    writer.write("# Attempt nr. " + (i + 1));
                    writer.newLine();
                    writer.write("#");
                    writer.newLine();
                    for (String line : previousLogs.get(i).split("\n")) {
                        writer.write(line);
                        writer.newLine();
                    }
                }
                writer.write("#");
                writer.newLine();
                writer.write("# Attempt nr. " + (previousLogs.size() + 1));
                writer.newLine();
                writer.write("#");
                writer.newLine();
                if (currentLog != null) {
                    for (String line : currentLog.split("\n")) {
                        writer.write(line);
                        writer.newLine();
                    }
                }
            } else if (currentLog != null) {
                for (String line : currentLog.split("\n")) {
                    writer.write(line);
                    writer.newLine();
                }
            }
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return downloadResource.register(to, "text/plain");
    }

    public static String createAuditReport(DownloadResource downloadResource, List<RolePrincipalPermissionRow> globalAuditReport,
                                           List<AuditPermissionRoleRow> folderAuditReport) {
        final DateTimeFormatter auditDateFormat = DateTimeFormat.forPattern("yyyy-MM-dd_hhmmss");
        final String serverName = ServerConfiguration.getInstance().getServerUrl();
        final String deployVersion = ReleaseInfo.getReleaseInfo().getVersion();
        return createReportToken(AuditReportUtility.createAuditReport(globalAuditReport, folderAuditReport, serverName, deployVersion), downloadResource, auditDateFormat);
    }

    public static String createReportToken(Workbook workbook, DownloadResource downloadResource, DateTimeFormatter dateFormat) {
        String fileName = format("%s - Deploy.xlsx", dateFormat.print(LocalDateTime.now()));
        File to = new File(WorkdirHolder.get().getPath(), fileName);
        try (FileOutputStream fos = new FileOutputStream(to)) {
            workbook.write(fos);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return downloadResource.register(to, "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml");
    }

    public static String createDeploymentsDashboardReport(DateTime begin, DateTime end,
                                                Collection<StatusOverviewReportLine> dashboardReport,
                                                Collection<StatusOverviewDataReportLine> deploymentReport,
                                                DownloadResource downloadResource) {
        final DateTimeFormatter reportDateFormat = DateTimeFormat.forPattern("yyyy-MM-dd_hhmmss");
        final String serverName = ServerConfiguration.getInstance().getServerUrl();
        final String deployVersion = ReleaseInfo.getReleaseInfo().getVersion();
        return createReportToken(DeploymentReportsUtility.createDeploymentsDashboardReport(begin, end, dashboardReport,
                deploymentReport, serverName, deployVersion), downloadResource, reportDateFormat);
    }
}
