package com.xebialabs.deployit.core.rest.api.reports.mapper;

import ai.digital.deploy.sql.model.ReportLine;
import com.google.common.base.Joiner;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

public class ReportToCsvMapper {
    private final RowMapper rowMapper;

    public ReportToCsvMapper(final Stream<ReportLine> reportLines) {
        rowMapper = new ImplicitRowMapper(reportLines, null);
    }

    public ReportToCsvMapper(final Stream<ReportLine> reportLines, final Collection<String> fields, final Map<String,String> headersDescMap) {
        if (fields == null) {
            rowMapper = new ImplicitRowMapper(reportLines, headersDescMap);
        } else {
            rowMapper = new ExplicitRowMapper(reportLines, sortFields(fields),headersDescMap);
        }
    }

    private List<String> sortFields(final Collection<String> fields) {
        return fields.stream().sorted().collect(Collectors.toList());
    }

    public Stream<String> getCsv() {
        return rowMapper.getRows();
    }

    private abstract static class RowMapper {
        protected static final Joiner LINE_JOINER = Joiner.on("\n");
        protected static final Joiner FIELD_JOINER = Joiner.on(",");
        protected static final Function<Object, @Nullable String> QUOTE = (final Object input) -> String.format("\"%s\"", input);

        protected final Stream<ReportLine> reportLines;
        protected final Map<String,String> headersDescMap;

        public abstract Stream<String> getRows();

        RowMapper(final Stream<ReportLine> reportLines, final Map<String,String> headersDescMap) {
            this.reportLines = reportLines;
            this.headersDescMap = headersDescMap;
        }

        protected String line(final Iterable<?> fields) {
            return FIELD_JOINER.join(StreamSupport.stream(fields.spliterator(), false).map(QUOTE).collect(Collectors.toList()));
        }
    }

    private static class ImplicitRowMapper extends RowMapper {
        private boolean first = true;


        ImplicitRowMapper(final Stream<ReportLine> reportLines, final Map<String,String> headersDescMap) {
            super(reportLines,headersDescMap);
        }

        @Override
        public Stream<String> getRows() {
            return reportLines.map(this::getLine);
        }

        private String getLine(final ReportLine reportLine) {
            TreeMap<String, Object> sorted = sorted(reportLine);
            if (first) {
                first = false;
                return LINE_JOINER.join(
                        line(getHeaderDesc(sorted,headersDescMap)),
                        line(sorted.values())
                );
            } else {
                return line(sorted.values());
            }

        }

        private static @NotNull List<String> getHeaderDesc(Map<String, Object> sorted, Map<String,String> headersMap) {
            if (headersMap == null) {
                return sorted.keySet().stream().toList();
            } else {
                return sorted.keySet().stream()
                        .map(key -> headersMap.getOrDefault(key, key))
                        .collect(Collectors.toList());
            }
        }

        private static TreeMap<String, Object> sorted(final ReportLine line) {
            return new TreeMap<>(line.getValues());
        }

    }

    private static class ExplicitRowMapper extends RowMapper {
        private final List<String> fields;

        ExplicitRowMapper(final Stream<ReportLine> reportLines, final List<String> fields, final Map<String,String> headersMap) {
            super(reportLines, headersMap);
            this.fields = fields;
        }

        @Override
        public Stream<String> getRows() {
            return Stream.concat(Stream.of(getHeader()), reportLines.map(this::getLine));
        }

        private String getHeader() {
            return line(getHeaders());
        }

        private List<String> getHeaders() {
            if (headersDescMap == null) {
                return fields;
            }
            return fields.stream()
                    .map(field -> headersDescMap.getOrDefault(field, field))
                    .collect(Collectors.toList());
        }

        private String getLine(final ReportLine reportLine) {
            Map<String, Object> values = reportLine.getValues();
            List<Object> sortedValues = this.fields.stream().map(values::get).collect(Collectors.toList());
            return line(sortedValues);
        }
    }
}
