package com.xebialabs.deployit.core.service.impl

import ai.digital.deploy.sql.model.ConfigurationDto
import ai.digital.deploy.settings.GeneralSettings
import com.xebialabs.deployit.core.service.GeneralSettingsService
import com.xebialabs.deployit.core.util.GeneralSettingsUtils.getLogoMimeType
import com.xebialabs.deployit.repository.ConfigurationRepository
import org.jsoup.Jsoup
import org.jsoup.safety.Safelist
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.nio.charset.StandardCharsets
import java.util.Base64

@Service
class GeneralSettingsServiceImpl(@Autowired configurationRepository: ConfigurationRepository) extends GeneralSettingsService {
  private val HEADER_COLOR = "header_color"
  private val INSTANCE_NAME = "instance_name"
  private val LOGIN_SCREEN_MESSAGE = "login_screen_message"
  private val LOGO_FILE = "logo_file"
  private val PAT_ENABLE_FOR_SSO = "pat_enable_for_sso"
  private val PAT_EXPIRATION_DURATION = "pat_expiration_duration"
  private val PAT_EMAIL_NOTIFICATION_ENABLED = "pat_email_notification_enabled"
  private val PAT_EMAIL_NOTIFICATION_TRIGGER_PERIOD = "pat_email_notification_trigger_period"
  private val PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_REF = "pat_email_notification_smtp_server_ci_ref"
  private val PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_NAME = "pat_email_notification_smtp_server_ci_name"

  private val svgSafeList: Safelist = Safelist.basic()
    .addTags("svg", "g", "path", "rect", "circle", "ellipse", "line", "polyline", "polygon", "text", "tspan")
    .addAttributes(":all", "style", "fill", "fill-rule", "clip-rule", "stroke", "d", "width", "height", "viewBox",
      "x", "y", "r", "cx", "cy", "points", "transform", "xmlns")
    .addProtocols("xlink:href", "http", "https")

  override def getGeneralSettings: GeneralSettings = {
    var headerColor: String = ""
    var instanceName: String = ""
    var loginScreenMessage: String = ""
    var logoFile: String = ""
    var patEnableForSSO: Boolean = false
    var patExpirationDuration: String = ""
    var patEmailNotificationEnabled: Boolean = false
    var patEmailNotificationTriggerPeriod: String = ""
    var patEmailNotificationSMTPServerCiRef: String = ""
    var patEmailNotificationSMTPServerCiName: String = ""

    val configurationList = configurationRepository.getConfigurationByKey
    configurationList.foreach {
      case conf if conf.key.endsWith(HEADER_COLOR) => headerColor = conf.value
      case conf if conf.key.endsWith(INSTANCE_NAME) => instanceName = conf.value
      case conf if conf.key.endsWith(LOGIN_SCREEN_MESSAGE) => loginScreenMessage = conf.value
      case conf if conf.key.endsWith(LOGO_FILE) => logoFile = conf.value
      case conf if conf.key.endsWith(PAT_ENABLE_FOR_SSO) => patEnableForSSO = conf.value.toBoolean
      case conf if conf.key.endsWith(PAT_EXPIRATION_DURATION) => patExpirationDuration = conf.value
      case conf if conf.key.endsWith(PAT_EMAIL_NOTIFICATION_ENABLED) => patEmailNotificationEnabled = conf.value.toBoolean
      case conf if conf.key.endsWith(PAT_EMAIL_NOTIFICATION_TRIGGER_PERIOD) => patEmailNotificationTriggerPeriod = conf.value
      case conf if conf.key.endsWith(PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_REF) => patEmailNotificationSMTPServerCiRef = conf.value
      case conf if conf.key.endsWith(PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_NAME) => patEmailNotificationSMTPServerCiName = conf.value
    }
    GeneralSettings(headerColor, instanceName, loginScreenMessage, logoFile, patEnableForSSO, patExpirationDuration,
      patEmailNotificationEnabled, patEmailNotificationTriggerPeriod, patEmailNotificationSMTPServerCiRef, patEmailNotificationSMTPServerCiName)
  }

  override def updateGeneralSettings(generalSettings: GeneralSettings): GeneralSettings = {
    val sanitizedGeneralSettings =  if (generalSettings.logoFile != null && generalSettings.logoFile.nonEmpty) {
      val logoMimeType = getLogoMimeType(generalSettings.logoFile)
      val sanitizedLogoFile = sanitizeLogoFile(logoMimeType, generalSettings.logoFile)
      generalSettings.copy(logoFile = sanitizedLogoFile)
    } else {
      generalSettings
    }
    val keysList: Set[String] = Set(HEADER_COLOR, INSTANCE_NAME, LOGIN_SCREEN_MESSAGE, LOGO_FILE, PAT_ENABLE_FOR_SSO, PAT_EXPIRATION_DURATION,
      PAT_EMAIL_NOTIFICATION_ENABLED, PAT_EMAIL_NOTIFICATION_TRIGGER_PERIOD, PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_REF, PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_NAME)

    keysList.foreach(key => {
      var value: String = ""
      key match {
        case HEADER_COLOR => value = generalSettings.headerColor
        case INSTANCE_NAME => value = generalSettings.instanceName
        case LOGIN_SCREEN_MESSAGE => value = sanitizeHtml(generalSettings.loginScreenMessage)
        case LOGO_FILE => value = sanitizedGeneralSettings.logoFile
        case PAT_ENABLE_FOR_SSO => value = generalSettings.patEnableForSSO.toString
        case PAT_EXPIRATION_DURATION => value = generalSettings.patExpirationDuration
        case PAT_EMAIL_NOTIFICATION_ENABLED => value = generalSettings.patEmailNotificationEnabled.toString
        case PAT_EMAIL_NOTIFICATION_TRIGGER_PERIOD => value = generalSettings.patEmailNotificationTriggerPeriod
        case PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_REF => value = generalSettings.patEmailNotificationSMTPServerCiRef
        case PAT_EMAIL_NOTIFICATION_SMTP_SERVER_CI_NAME => value = generalSettings.patEmailNotificationSMTPServerCiName
      }
      configurationRepository.saveOrUpdate(ConfigurationDto(key.replace("_", " "), key, value))
    })
    getGeneralSettings
  }

  private def sanitizeLogoFile(contentType: String, logoFile: String): String = {
    val sanitizedContent = if (contentType == "image/svg+xml") {
      val base64Index = logoFile.indexOf("base64,") + "base64,".length
      val base64Content = logoFile.substring(base64Index)
      val decodedBytes = Base64.getDecoder.decode(base64Content)
      val imageContent = sanitizeSvg(new String(decodedBytes, StandardCharsets.UTF_8))
      val sanitizedBytes = imageContent.getBytes(StandardCharsets.UTF_8)
      val sanitizedBase64Content = Base64.getEncoder.encodeToString(sanitizedBytes)
      logoFile.substring(0, base64Index) + sanitizedBase64Content
    } else {
      logoFile
    }
    sanitizedContent
  }

  private def sanitizeSvg(svgContent: String): String = {
    Jsoup.clean(svgContent, "", svgSafeList, new org.jsoup.nodes.Document.OutputSettings().prettyPrint(false))
  }

  private def sanitizeHtml(html: String): String = {
    Jsoup.clean(html, Safelist.basic())
  }
}
