package com.xebialabs.deployit.core.rest.json;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.xebialabs.deployit.engine.api.dto.XLDAsCodeResult;
import org.springframework.stereotype.Component;

import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.WebApplicationException;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.ext.MessageBodyReader;
import jakarta.ws.rs.ext.MessageBodyWriter;
import jakarta.ws.rs.ext.Provider;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;

/**
 * JAX-RS MessageBodyReader and MessageBodyWriter for XLDAsCodeResult.
 * Enables JSON serialization/deserialization of XLDAsCodeResult DTOs between
 * master and external worker processes.
 */
@Component
@Provider
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
public class XLDAsCodeResultJsonReaderWriter implements MessageBodyReader<XLDAsCodeResult>, MessageBodyWriter<XLDAsCodeResult> {

    private final ObjectMapper objectMapper = new ObjectMapper();

    @Override
    public boolean isReadable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return XLDAsCodeResult.class.isAssignableFrom(type);
    }

    @Override
    public XLDAsCodeResult readFrom(Class<XLDAsCodeResult> type, Type genericType,
                                    Annotation[] annotations, MediaType mediaType,
                                    MultivaluedMap<String, String> httpHeaders, InputStream entityStream)
            throws IOException, WebApplicationException {
        return objectMapper.readValue(entityStream, XLDAsCodeResult.class);
    }

    @Override
    public boolean isWriteable(Class<?> type, Type genericType, Annotation[] annotations, MediaType mediaType) {
        return XLDAsCodeResult.class.isAssignableFrom(type);
    }

    @Override
    public long getSize(XLDAsCodeResult result, Class<?> type, Type genericType,
                        Annotation[] annotations, MediaType mediaType) {
        return -1L; // unknown size
    }

    @Override
    public void writeTo(XLDAsCodeResult result, Class<?> type, Type genericType,
                       Annotation[] annotations, MediaType mediaType, MultivaluedMap<String, Object> httpHeaders,
                       OutputStream entityStream) throws IOException, WebApplicationException {
        byte[] json = objectMapper.writeValueAsBytes(result);
        entityStream.write(json);
    }
}
