package com.xebialabs.deployit.core.rest.api;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.core.rest.api.DeploymentResource.DeploymentType.INITIAL;
import static com.xebialabs.deployit.core.rest.api.DeploymentResource.DeploymentType.UNDEPLOYMENT;
import static com.xebialabs.deployit.core.rest.api.DeploymentResource.DeploymentType.UPGRADE;
import static com.xebialabs.deployit.repository.JcrPathHelper.getParentId;
import static javax.ws.rs.core.Response.Status.CREATED;

import java.util.Collection;
import java.util.List;

import javax.ws.rs.core.Response;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.Mappings;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.core.api.DeploymentProxy;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.RepositoryObjects;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.reflect.ConfigurationItemDescriptor;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.task.ExecutionEngine;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskStep;
import com.xebialabs.deployit.typedescriptor.ConfigurationItemDescriptorRepositoryHolder;

@Controller
public class DeploymentResource extends AbstractSecuredResource implements DeploymentProxy {

	@Autowired
	private RepositoryService repositoryService;

	@Autowired
	private DeploymentService deploymentService;

	@Autowired
	private DtoConverter dtoConverter;

	@Autowired
	private ExecutionEngine engine;

	@Autowired
	private Validator validator;

	enum DeploymentType {
		INITIAL, UPGRADE, UNDEPLOYMENT
	}

	@Override
	public Response generateMappings(String sourceId, String targetId) {
		ConfigurationItemEntity source = repositoryService.readFully(sourceId);
		ConfigurationItemEntity target = targetId != null ? (ConfigurationItemEntity) repositoryService.readFully(targetId) : null;
		DeploymentType type = determineType(source, target);

		Mappings mappings = new Mappings();
		if (type == INITIAL) {
			logger.info("Initial deployment of {} to {}", sourceId, targetId);
			mappings = deploymentService.generateDefaultMappings(source, target);
			return ResponseFactory.status(CREATED).entity(dtoConverter.toDto(mappings)).build();
		} else if (type == UPGRADE) {
			logger.info("Upgrade deployment of {} to {}", sourceId, targetId);
			mappings = deploymentService.generateMappingsForUpgradeDeployment(source, target);
		} else if (type == UNDEPLOYMENT) {
			// no mappings are needed, return empty mappings
		} else {
			throw new Checks.IncorrectArgumentException("Can't generate mappings for deployment type %s", type);
		}

		return ResponseFactory.status(CREATED).entity(dtoConverter.toDto(mappings)).build();
	}

	@Override
	public Response generatePartialMappings(final List<String> partialSourceIds, final String targetId, final String mappingType) {
		checkArgument(partialSourceIds.size() > 0, "Should have at least on source to generate mappings for.");

		String deploymentPackageId = null;
		final List<ConfigurationItemEntity> sourceCIs = Lists.newArrayList();
		for (String id : partialSourceIds) {
			sourceCIs.add((ConfigurationItemEntity) repositoryService.readFully(id));
			// N.B.: it is assumed here that package members are subnodes of the package they belong to.
			String thisDeploymentPackageId = getParentId(id);
			if (deploymentPackageId == null) {
				deploymentPackageId = thisDeploymentPackageId;
			} else {
				checkArgument(StringUtils.equals(deploymentPackageId, deploymentPackageId), "All sources should be from same package");
			}
		}

		ConfigurationItemEntity deploymentPackage = repositoryService.readFully(deploymentPackageId);
		ConfigurationItemEntity target = repositoryService.readFully(targetId);

		DeploymentType type = determineType(sourceCIs.get(0), target);
		if (type == INITIAL) {
			return ResponseFactory.created(
			        dtoConverter.toDto(deploymentService.generatePartialDefaultMappings(deploymentPackage, sourceCIs, target, mappingType))).build();
		} else {
			throw new Checks.IncorrectArgumentException("Target of partial mappings should be an environment or an environment member.");
		}
	}

	@Override
	public Response prepare(String sourceId, String targetId, RepositoryObjects mappings) {
		Task deploymentTask = createDeploymentTask(sourceId, targetId, mappings);
		String taskId = engine.register(deploymentTask);
		Steps steps = dtoConverter.toDto(deploymentTask);
		steps.setTaskId(taskId);
		return ResponseFactory.ok(steps).build();
	}

	@Override
	public Response validate(final String sourceId, final String targetId, final RepositoryObjects mappings) {
		if (!validateMappingsWithValidator(mappings)) {
			return ResponseFactory.badRequest(mappings).build();
		}

		final Task task = createDeploymentTask(sourceId, targetId, mappings);
		try {
			final List<TaskStep> taskSteps = task.getSteps();
			if (taskSteps.isEmpty()) {
				throw new InvalidMappingsException("The mappings did not lead to any steps, no action will be taken.");
			}
		} finally {
			task.destroy();
		}
		return ResponseFactory.noContent().build();
	}

	private Task createDeploymentTask(final String sourceId, final String targetId, final RepositoryObjects mappings) {
		ConfigurationItemEntity source = repositoryService.readFully(sourceId);
		ConfigurationItemEntity target = targetId != null ? (ConfigurationItemEntity) repositoryService.readFully(targetId) : null;

		final DeploymentType type = determineType(source, target);
		checkPermissions(source, target, type);
		Task deploymentTask = null;
		if (type == INITIAL) {
			Collection<ConfigurationItemEntity> mappingEntities = dtoConverter.fromDto(mappings);
			deploymentTask = deploymentService.prepareInitialDeployment(source, target, mappingEntities);
		} else if (type == UPGRADE) {
			Collection<ConfigurationItemEntity> mappingEntities = dtoConverter.fromDto(mappings);
			deploymentTask = deploymentService.prepareUpgradeDeployment(source, target, mappingEntities);
		} else if (type == UNDEPLOYMENT) {
			deploymentTask = deploymentService.prepareUndeployment(source);
		} else {
			throw new UnsupportedOperationException("type " + type + " is not supported yet.");
		}
		return deploymentTask;
	}

	private void checkPermissions(final ConfigurationItemEntity source, final ConfigurationItemEntity target, final DeploymentType type) {
		switch (type) {
		case INITIAL:
			checkPermission(Permission.DEPLOY_INITIAL, newArrayList(target.getId()));
			break;
		case UPGRADE:
			// An upgrade is source=DeploymentPackage, target=Deployment. We need to check whether you have rights
			// to deploy the upgrade to the target Environment, which is the target of the deployment.
			checkPermission(Permission.DEPLOY_UPGRADE, newArrayList((String) target.getValue("target")));
			break;
		case UNDEPLOYMENT:
			checkPermission(Permission.UNDEPLOY, newArrayList((String) source.getValue("target")));
			break;
		default:
			throw new IllegalStateException("Cannot do deployment of unknown type.");
		}
	}

	private boolean validateMappingsWithValidator(final RepositoryObjects mappings) {
		boolean allValid = true;
		for (RepositoryObject mapping : mappings.getObjects()) {
			final RepositoryObjectEntity mappingEntity = dtoConverter.fromDto(mapping);
			final Validator.Validations validations = validator.validate(mappingEntity);
			if (validations.hasMessages()) {
				mapping.setValidations(dtoConverter.toDto(validations));
				allValid = false;
			}
		}
		return allValid;
	}

	/**
	 * Types: - INITIAL: source = DeploymentPackage; target = Environment - UPGRADE: source = (new) DeploymentPackage; target = Deployment - REDEPLOYMENT:
	 * source = same DeploymentPackage; target = Deployment (TODO ??) - PROMOTION: source = Deployment; target = (different) Environment - UNDEPLOYMENT: source
	 * = Deployment; target = <b>null</b>
	 * 
	 * @param source
	 * @param target
	 * @return
	 */
	private DeploymentType determineType(final ConfigurationItemEntity source, final ConfigurationItemEntity target) {
		final String sourceType = source.getConfigurationItemTypeName();
		final ConfigurationItemDescriptor sourceDescriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(sourceType);
		if (isAssignable(sourceDescriptor, Deployment.class) && target == null) {
			return UNDEPLOYMENT;
		} else if (target == null) {
			throw new Checks.IncorrectArgumentException("A target should be given unless it is an undeployment.");
		}

		final String targetType = target.getConfigurationItemTypeName();
		final ConfigurationItemDescriptor targetDescriptor = ConfigurationItemDescriptorRepositoryHolder.getDescriptor(targetType);
		if (isAssignable(targetDescriptor, Deployment.class)) {
			return UPGRADE;
		} else {
			return INITIAL;
		}
	}

	private boolean isAssignable(ConfigurationItemDescriptor desc, Class<?> clazz) {
		return clazz.isAssignableFrom(desc.getTypeClass());
	}

	@SuppressWarnings("serial")
	@HttpResponseCodeResult(statusCode = 400)
	public static class DeploymentException extends DeployitException {
		public DeploymentException(final String messageTemplate, final Object... params) {
			super(messageTemplate, params);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(DeploymentResource.class);

	@SuppressWarnings("serial")
	@HttpResponseCodeResult(statusCode = 406)
	private class InvalidMappingsException extends DeployitException {
		public InvalidMappingsException(final String s) {
			super(s);
		}
	}
}
