package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.api.ImportablePackageProxy;
import com.xebialabs.deployit.core.api.dto.FileUpload;
import com.xebialabs.deployit.core.api.dto.ImportablePackages;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.importer.ImporterService;
import com.xebialabs.deployit.service.importer.PackageInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.core.Response;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;

/**
 * I am A REST resource that exposes functionality on the Deployable packages.
 * <p/>
 * Functionality includes: - Listing all Deployment packages in the importablePackages directory on the server.
 */
@Controller
public class ImportablePackageResource extends AbstractSecuredResource implements ImportablePackageProxy {

	@Autowired
	private ImporterService importerService;

	@Autowired
	private RepositoryService repositoryService;

    @Autowired
    private DtoConverter dtoConverter;

	public Response list() {
		List<String> files = importerService.list();
		return ResponseFactory.ok(new ImportablePackages(files)).build();
	}

	public Response importPackage(String file) {
		final PackageInfo packageInfo = importerService.preparePackage(file);
		return ResponseFactory.created(checkAndDoImport(packageInfo)).build();
	}

    @Override
    public Response importPackageFromUrl(String url) {
        final PackageInfo packageInfo;
        try {
            packageInfo = importerService.preparePackage(new URL(url));
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
        return ResponseFactory.created(checkAndDoImport(packageInfo)).build();
    }

    @Override
	public Response importUploadedPackage(FileUpload form) {
		byte[] uploadedPackage = form.getFileData();
		final PackageInfo packageInfo = importerService.preparePackage(uploadedPackage);
		return ResponseFactory.created(checkAndDoImport(packageInfo)).build();
	}

	private RepositoryObject checkAndDoImport(final PackageInfo packageInfo) {
		try {
            logger.debug("Going to import {} version {}", packageInfo.getApplicationId(), packageInfo.getApplicationVersion());
			if (repositoryService.checkNodeExists(packageInfo.getApplicationId())) {
				checkPermission(Permission.IMPORT_UPGRADE, newArrayList(packageInfo.getApplicationId()));
			} else {
				checkPermission(Permission.IMPORT_INITIAL);
			}
			String id = importerService.importPackage(packageInfo);
            final RepositoryObjectEntity objectEntity = repositoryService.read(id);
            return dtoConverter.toDto(objectEntity);
        } finally {
			importerService.cleanUp(packageInfo);
		}
	}

    private static final Logger logger = LoggerFactory.getLogger(ImportablePackageResource.class);
}
