package com.xebialabs.deployit.core.rest.api.reports;

import static com.xebialabs.deployit.task.Task.DEPLOYMENT_PREFIX;
import static com.xebialabs.deployit.task.Task.UPGRADE_PREFIX;
import static com.xebialabs.deployit.task.Task.State.DONE;
import static com.xebialabs.deployit.task.Task.State.STOPPED;

import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang.StringUtils;

import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskRegistry;
import com.xebialabs.deployit.task.Task.State;
import com.xebialabs.deployit.task.labelparser.DeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.InitialDeploymentTaskLabelParser;
import com.xebialabs.deployit.task.labelparser.UpgradeTaskLabelParser;

public class DeploymentsStateBreakdownPerEnvironmentReport {

	private TaskRegistry taskRegistry;

	public DeploymentsStateBreakdownPerEnvironmentReport(TaskRegistry taskRegistry) {
		this.taskRegistry = taskRegistry;
	}

	public Report report(final List<String> repositoryEntityIds, final Date startDate, final Date endDate) {
		Report report = new Report();
		Map<String, TaskStateData> tasksMap = Maps.newHashMap();
		final Collection<Task> filteredTasks = Collections2.filter(taskRegistry.getAllArchivedTasks(),
				new Predicate<Task>() {
					@Override
					public boolean apply(final Task input) {
						State state = input.getState();
						final Calendar completionDate = input.getCompletionDate();
						boolean isValidState = ((state.equals(DONE) || state.equals(STOPPED)));
						if (!isValidState || !input.isDeploymentTask() || completionDate == null) {
							return false;
						}
						String targetEntityId = getTargetEntityId(input);
						if (StringUtils.isEmpty(targetEntityId) || !repositoryEntityIds.contains(targetEntityId)) {
							return false;
						}
						boolean isAfterBegin = completionDate.after(startDate.getCalendar());
						boolean isEqualsBegin = completionDate.equals(startDate.getCalendar());
						boolean isBeforeEnd = completionDate.before(endDate.getCalendar());
						return (isAfterBegin || isEqualsBegin) && isBeforeEnd;
					}
				});

		for (Task task : filteredTasks) {
			createTasksMap(tasksMap, task);
		}
		
		for (Iterator<String> iterator = repositoryEntityIds.iterator(); iterator.hasNext();) {
			String environmentID  = (String) iterator.next();
			environmentID = environmentID.substring(environmentID.indexOf("/") + 1);
			if(!tasksMap.containsKey(environmentID)){
				tasksMap.put(environmentID, new TaskStateData(new AtomicInteger(0), new AtomicInteger(0)));
			}
		}
		report = generateReportDataFromTaskMap(tasksMap);
		return report;
	}

	private Report generateReportDataFromTaskMap(Map<String, TaskStateData> tasksMap) {
		final Report report = new Report();
		for (Map.Entry<String, TaskStateData> entry : tasksMap.entrySet()) {
			final Report.ReportLine reportLine = report.addLine();
			reportLine.addValue("environmentName", entry.getKey());
			reportLine.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
			reportLine.addValue("noOfFailedDeployments", entry.getValue().getNoOfFailedDeployments().toString());
		}
		return report;
	}

	private void createTasksMap(Map<String, TaskStateData> tasksMap, Task task) {
		String targetEntityId = getTargetEntityId(task);
		targetEntityId = targetEntityId.substring(targetEntityId.indexOf("/") + 1);
		if (task.getState().equals(DONE)) {
			if (!tasksMap.containsKey(targetEntityId)) {
				tasksMap.put(targetEntityId, new TaskStateData(new AtomicInteger(1), new AtomicInteger(0)));
			} else {
				tasksMap.get(targetEntityId).getNoOfSuccessfulDeployments().incrementAndGet();
			}
		} else if (task.getState().equals(STOPPED)) {
			if (!tasksMap.containsKey(targetEntityId)) {
				tasksMap.put(targetEntityId, new TaskStateData(new AtomicInteger(0), new AtomicInteger(1)));
			} else {
				tasksMap.get(targetEntityId).getNoOfFailedDeployments().incrementAndGet();
			}
		}
	}

	private String getTargetEntityId(Task inputTask) {
		String targetEntityId = "";
		DeploymentTaskLabelParser labelParser = null;
		String taskLabel = inputTask.getLabel();
		if (taskLabel.startsWith(DEPLOYMENT_PREFIX)) {
			labelParser = new InitialDeploymentTaskLabelParser(taskLabel);
		} else if (taskLabel.startsWith(UPGRADE_PREFIX)) {
			labelParser = new UpgradeTaskLabelParser(taskLabel);
		}
		if (labelParser != null) {
			targetEntityId = labelParser.getEnvironment();
		}
		return targetEntityId;
	}

}
