package com.xebialabs.deployit.core.rest.api;


import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.DeploymentProxy;
import com.xebialabs.deployit.core.api.dto.ConfigurationItem;
import com.xebialabs.deployit.core.api.dto.Deployment;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.repository.ConfigurationItemEntity;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.deployment.DeployedService;
import com.xebialabs.deployit.service.deployment.DeploymentService;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.task.ExecutionEngine;
import com.xebialabs.deployit.task.Task;
import com.xebialabs.deployit.task.TaskStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;
import java.util.Collection;
import java.util.List;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;

@Controller
public class DeploymentResource extends AbstractSecuredResource implements DeploymentProxy {

	@Autowired
	private RepositoryService repositoryService;

	@Autowired
	private DeploymentService deploymentService;

	@Autowired
	private DeployedService deployedService;

	@Autowired
	private DtoConverter dtoConverter;

	@Autowired
	private ExecutionEngine engine;

	@Autowired
	private Validator validator;

	@Override
	public Response prepareInitial(@QueryParam("version") String versionId, @QueryParam("environment") String environmentId) {
		checkPermission(Permission.DEPLOY_INITIAL, newArrayList(environmentId));
		checkNotNull(versionId, "version");
		checkNotNull(environmentId, "environment");
		ConfigurationItemEntity version = repositoryService.readFully(versionId);
		ConfigurationItemEntity environment = repositoryService.readFully(environmentId);

		checkArgument(DescriptorRegistry.getDescriptor(version.getType()).isAssignableTo(Version.class), "%s is not a Version", versionId);
		checkArgument(DescriptorRegistry.getDescriptor(environment.getType()).isAssignableTo(Environment.class), "%s is not an Environment", environmentId);

		Deployment deployment = createDeployment(version, environmentId);

		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response prepareUpgrade(@QueryParam("version") String versionId, @QueryParam("deployedApplication") String deployedApplicationId) {
		checkNotNull(versionId, "version");
		checkNotNull(deployedApplicationId, "deployedApplication");
		ConfigurationItemEntity deployedApplication = repositoryService.readFully(deployedApplicationId);
		checkArgument(deployedApplication.getType().equals(Type.valueOf(DeployedApplication.class)), "%s is not a DeployedApplication", deployedApplicationId);

		String environmentId = (String) deployedApplication.getValue("environment");
		checkPermission(Permission.DEPLOY_UPGRADE, environmentId);

		ConfigurationItemEntity version = repositoryService.readFully(versionId);
		checkArgument(DescriptorRegistry.getDescriptor(version.getType()).isAssignableTo(Version.class), "%s is not a Version", versionId);

		ListMultimap<Boolean,ConfigurationItemEntity> upgradedDeployeds = deployedService.generateUpgradedDeployeds(version, deployedApplication);

		Deployment deployment = createDeployment(version, environmentId);
		deployment.setUpgrade(true);
		deployment.setDeployeds(dtoConverter.deployedsToDto(upgradedDeployeds));

		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response generateAllDeployeds(Deployment deployment) {
		checkNotNull(deployment, "deployment");
		checkPermission(Permission.DEPLOY_INITIAL, deployment.getEnvironment());

		ConfigurationItemEntity version = repositoryService.readFully(deployment.getVersion());
		ConfigurationItemEntity environment = repositoryService.readFully(deployment.getEnvironment());

		ListMultimap<Boolean, ConfigurationItemEntity> initialDeployeds = deployedService.generateAllDeployeds(version, environment);

		deployment.addAll(dtoConverter.deployedsToDto(initialDeployeds));
		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response generateSelectedDeployeds(@QueryParam("deployables") List<String> deployableIds, Deployment deployment) {
		checkNotNull(deployment, "deployment");
		checkPermission(Permission.DEPLOY_INITIAL, deployment.getEnvironment());
		checkArgument(deployableIds.size() > 0, "Should select at least one deployable to generate a deployed");

		ConfigurationItemEntity environment = repositoryService.readFully(deployment.getEnvironment());

		final List<ConfigurationItemEntity> deployableCIs = Lists.newArrayList();
		for (String id : deployableIds) {
			deployableCIs.add((ConfigurationItemEntity) repositoryService.readFully(id));
			checkArgument(id.startsWith(deployment.getVersion()), "All sources should be from same package");
		}

		ListMultimap<Boolean, ConfigurationItemEntity> selectedDeployeds = deployedService.generateSelectedDeployeds(deployableCIs, environment);
		deployment.addAll(dtoConverter.deployedsToDto(selectedDeployeds));
		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response generateSingleDeployed(@QueryParam("deployable") String deployableId, @QueryParam("container") String containerId, @QueryParam("deployedtype") String deployedType, Deployment deployment) {
		checkNotNull(deployment, "deployment");
		checkPermission(Permission.DEPLOY_INITIAL, deployment.getEnvironment());

		checkNotNull(deployableId, "deployable");
		checkNotNull(containerId, "container");

		ConfigurationItemEntity deployable = repositoryService.readFully(deployableId);
		ConfigurationItemEntity container = repositoryService.read(containerId);
		ConfigurationItemEntity environment = repositoryService.read(deployment.getEnvironment());

		ListMultimap<Boolean, ConfigurationItemEntity> deployeds = deployedService.generateSelectedDeployed(deployable, container, deployedType, environment);
		deployment.addAll(dtoConverter.deployedsToDto(deployeds));
		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response validate(Deployment deployment) {
		if (!validateDeployedsWithValidator(deployment.getDeployeds())) {
			return ResponseFactory.badRequest(deployment).build();
		}

		final Task task = createDeploymentTask(deployment);
		try {
			final List<TaskStep> taskSteps = task.getSteps();
			checkState(!taskSteps.isEmpty(), "The task did not deliver any steps.");
		} finally {
			task.destroy();
		}
		return ResponseFactory.ok(deployment).build();
	}

	@Override
	public Response deploy(Deployment deployment) {
		// TODO in case of an upgrade, we need to verify that a user on the CLI didn't break the security contract by manually adding extra deployeds...
		checkNotNull(deployment, "deployment");
		Task deploymentTask = createDeploymentTask(deployment);
		String taskId = engine.register(deploymentTask);
		Steps steps = dtoConverter.taskToDto(deploymentTask);
		steps.setTaskId(taskId);
		return ResponseFactory.ok(steps).build();
	}

	@Override
	public Response undeploy(String deployedApplicationId) {
		checkNotNull(deployedApplicationId, "deployedApplication");
		ConfigurationItemEntity deployedApplication = repositoryService.read(deployedApplicationId);
		checkPermission(Permission.UNDEPLOY, (String) deployedApplication.getValue("environment"));
		Task task = deploymentService.prepareUndeployment(deployedApplication);
		String taskId = engine.register(task);
		Steps steps = dtoConverter.taskToDto(task);
		steps.setTaskId(taskId);
		return ResponseFactory.ok(steps).build();
	}

	private Task createDeploymentTask(Deployment deployment) {
		ConfigurationItemEntity version = repositoryService.readFully(deployment.getVersion());

		Task deploymentTask = null;
		if (!deployment.isUpgrade()) {
			Collection<ConfigurationItemEntity> deployedEntities = dtoConverter.entitiesFromDto(deployment.getDeployeds());
			ConfigurationItemEntity environment = repositoryService.readFully(deployment.getEnvironment());
			deploymentTask = deploymentService.prepareInitialDeployment(version, environment, deployedEntities);
		} else {
			Collection<ConfigurationItemEntity> deployedEntities = dtoConverter.entitiesFromDto(deployment.getDeployeds());
			ConfigurationItemEntity existingDeployedApplication = repositoryService.readFully(constructDeployedApplicationId(version, deployment.getEnvironment()));
			deploymentTask = deploymentService.prepareUpgradeDeployment(version, existingDeployedApplication, deployedEntities);
		}
		return deploymentTask;
	}


	private boolean validateDeployedsWithValidator(final List<ConfigurationItem> deployeds) {
		boolean allValid = true;
        List<RepositoryObjectEntity> deployedEntitiesInCtx = newArrayList(dtoConverter.entitiesFromDto(deployeds));

        for (int i=0;i<deployeds.size();i++) {
            final ConfigurationItem deployed = deployeds.get(i);
            final RepositoryObjectEntity deployedEntity = deployedEntitiesInCtx.get(i);
            final Validator.Validations validations = validator.validate(deployedEntity, deployedEntitiesInCtx);
			if (validations.hasMessages()) {
				deployed.setValidations(dtoConverter.validationMessagesToDto(validations));
				allValid = false;
			}
		}
		return allValid;
	}

	private Deployment createDeployment(ConfigurationItemEntity version, String environmentId) {
		Deployment deployment = new Deployment();

		ConfigurationItemEntity deployedApplication = new ConfigurationItemEntity(Type.valueOf(DeployedApplication.class));
		deployedApplication.addValue("version", version.getId());
		deployedApplication.addValue("environment", environmentId);

		String id = constructDeployedApplicationId(version, environmentId);
		deployedApplication.setId(id);

		deployment.setDeployedApplication((ConfigurationItem) dtoConverter.entityToDto(deployedApplication));
		return deployment;
	}

	private String constructDeployedApplicationId(ConfigurationItemEntity version, String environmentId) {
		String applicationId = (String) version.getValue("application");
		int i = applicationId.lastIndexOf("/");
		return environmentId + applicationId.substring(i);
	}
}
