package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.RepositoryProxy;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.dto.RepositoryObjects;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.repository.RepositoryObjectEntity;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.validation.Validator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.core.Response;
import java.util.Collection;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static javax.ws.rs.core.Response.Status;
import static javax.ws.rs.core.Response.Status.BAD_REQUEST;

@Controller
public class RepositoryResource extends AbstractSecuredResource implements RepositoryProxy {

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private DtoConverter dtoConverter;

    @Autowired
    private Validator validator;

    @Override
    public Response create(String id, RepositoryObject dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.setId(id);
        RepositoryObjectEntity entity = dtoConverter.entityFromDto(dto);
        return createInternal(entity);
    }

    @Override
    public Response create(String id, ArtifactAndData dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.getArtifact().setId(id);
        RepositoryObjectEntity entity = dtoConverter.artifactEntityFromDto(dto);
        return createInternal(entity);
    }

    @Override
    public Response createMultiple(RepositoryObjects dto) {
	    checkPermission(Permission.EDIT_REPO, newArrayList(Collections2.transform(dto.getObjects(), new Function<RepositoryObject, String>() {

		    @Override
		    public String apply(RepositoryObject from) {
			    return from.getId();
		    }
	    })));
        Collection<RepositoryObjectEntity> entities = dtoConverter.entitiesFromDto(dto.getObjects());
        return createInternal(entities);
    }

    @Override
    public Response read(String id) {
        RepositoryObjectEntity entity = repositoryService.read(id);
        final RepositoryObject dto = dtoConverter.entityToDto(entity);
        return ResponseFactory.status(Status.OK).entity(dto).build();
    }

    @Override
    public Response update(String id, RepositoryObject dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.setId(id);
        RepositoryObjectEntity entity = dtoConverter.entityFromDto(dto);
        return updateInternal(entity);
    }

    @Override
    public Response update(String id, ArtifactAndData dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.getArtifact().setId(id);
        RepositoryObjectEntity entity = dtoConverter.artifactEntityFromDto(dto);
        return updateInternal(entity);
    }

    @Override
    public Response delete(String id) {
	    checkPermission(Permission.EDIT_REPO, id);
        repositoryService.delete(id);
        return ResponseFactory.noContent().build();
    }

    private Response createInternal(final RepositoryObjectEntity entity) {
        Response response = validateEntityAndCreateErrorResponseIfNeeded(entity);
        if (response != null) {
            return response;
        }

        repositoryService.create(entity);
        return reloadEntityAndCreateSuccessResponse(entity);
    }

	private Response validateEntityAndCreateErrorResponseIfNeeded(RepositoryObjectEntity entity) {
		Validator.Validations validations = validator.validate(entity, Lists.<RepositoryObjectEntity>newArrayList());
		if (validations.hasMessages()) {
		    return ResponseFactory.status(BAD_REQUEST).entity(dtoConverter.validationMessagesToDto(entity, validations)).build();
		}
		return null;
	}

    private Response validateEntityAndCreateErrorResponseIfNeeded(Collection<RepositoryObjectEntity> entities) {
		RepositoryObjects repositoryObjects = new RepositoryObjects();
		boolean allValid = true;
	    RepositoryObjectEntity[] entitiesArray = entities.toArray(new RepositoryObjectEntity[entities.size()]);
		for (RepositoryObjectEntity entity : entities) {
			Validator.Validations validations = validator.validate(entity, entitiesArray);
			repositoryObjects.add(dtoConverter.validationMessagesToDto(entity, validations));
			allValid = allValid && !validations.hasMessages();
		}
		if (!allValid) {
			return ResponseFactory.status(BAD_REQUEST).entity(repositoryObjects).build();
		}
        return null;
    }

    private Response reloadEntityAndCreateSuccessResponse(RepositoryObjectEntity entity) {
        RepositoryObjectEntity reloaded = repositoryService.read(entity.getId());
        return ResponseFactory.created(dtoConverter.entityToDto(reloaded)).build();
    }

    private Response createInternal(Collection<RepositoryObjectEntity> entities) {
        Response response = validateEntityAndCreateErrorResponseIfNeeded(entities);
        if (response != null) {
            return response;
        }

	    RepositoryObjectEntity[] entitiesArray = entities.toArray(new RepositoryObjectEntity[entities.size()]);
        repositoryService.create(entitiesArray);

        return reloadEntityAndCreateSuccessResponse(entities);
    }

    private Response reloadEntityAndCreateSuccessResponse(Collection<RepositoryObjectEntity> entities) {
        List<RepositoryObjectEntity> reloadedDtos = Lists.newArrayList();

        for (RepositoryObjectEntity entity : entities) {
            RepositoryObjectEntity reloadedEntity = repositoryService.read(entity.getId());
            reloadedDtos.add(reloadedEntity);
        }

        return ResponseFactory.created(dtoConverter.entitiesToDto(reloadedDtos)).build();
    }

    private Response updateInternal(final RepositoryObjectEntity entity) {
        Response response = validateEntityAndCreateErrorResponseIfNeeded(entity);
        if (response != null) {
            return response;
        }

        repositoryService.update(entity);
        return reloadEntityAndCreateSuccessResponse(entity);
    }
}
