package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.api.TaskProxy;
import com.xebialabs.deployit.core.api.dto.Steps;
import com.xebialabs.deployit.core.api.dto.TaskInfo;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.task.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.PathParam;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import java.util.Calendar;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.lang.String.format;

@Controller
public class TaskResource extends AbstractSecuredResource implements TaskProxy {

	@Autowired
	private DtoConverter dtoConverter;

	@Autowired
	private ExecutionEngine engine;

	@Autowired
	private TaskArchive taskArchive;

    @Override
	public Response start(final String taskId) {
		engine.execute(taskId);
		return ResponseFactory.accepted().build();
	}

    @Override
	public Response cancel(final String taskId) {
		engine.cancel(taskId);
		return ResponseFactory.status(Status.NO_CONTENT).build();
	}

    @Override
    public Response stop(final String taskId) {
        try {
            engine.stopExecution(taskId);
        } catch (IllegalStateException ise) {
            logger.error(format("Could not stop task %s", taskId), ise);
            return ResponseFactory.status(Response.Status.CONFLICT).entity(ise.getMessage()).build();
        }
        return ResponseFactory.accepted().build();
    }

    @Override
    public Response abort(final String taskId) {
        try {
            engine.abortExecution(taskId);
        } catch (IllegalStateException ise) {
            logger.error(format("Could not abort task %s", taskId), ise);
            return ResponseFactory.status(Response.Status.CONFLICT).entity(ise.getMessage()).build();
        }
        return ResponseFactory.accepted().build();
    }

    @Override
	public Response getTaskInfo(final String taskId) {
        Task runningTask = engine.getTask(taskId);
        TaskInfo taskInfoDto;
        if (runningTask != null) {
            taskInfoDto = dtoConverter.taskToDto(taskId, runningTask);
        } else {
            taskInfoDto = dtoConverter.archivedTaskToDto(taskArchive.getTask(taskId));
        }
        return ResponseFactory.ok(taskInfoDto).build();
	}

    @Override
    public Response toggleSkipSteps(@PathParam("taskid") final String taskId, @PathParam("steps") final String stepIds) {
	    checkPermission(Permission.TASK_SKIPSTEP);
        final Task task = engine.getTask(taskId);
	    checkArgument(task != null, "Could not find active task %s", taskId);
        String[] taskStepIds = stepIds.split(",");
        for (String stepId : taskStepIds) {
            final int id = Integer.parseInt(stepId);
            final TaskStep taskStep = task.getStep(id);
            if (taskStep.canSkip()) {
                logger.info("Skipping step :{}", taskStep.getDescription());
                taskStep.skip();
            } else if (taskStep.isSkipped()) {
                taskStep.unskip();
            }
        }

        return ResponseFactory.ok(dtoConverter.taskToDto(task)).build();
    }

	@Override
	public Response moveStep(@PathParam("taskid") String taskId, @PathParam("stepNr") int stepNr, @PathParam("position") int newPosition) {
		checkPermission(Permission.TASK_MOVESTEP);
		final Task task = engine.getTask(taskId);
		checkArgument(task != null, "Could not find active task %s", taskId);

		task.moveStep(stepNr, newPosition);

		return ResponseFactory.ok(dtoConverter.taskToDto(task)).build();
	}

	@Override
	public Response getStepInfo(String taskId, int stepNr, Date ifModifiedSince) {
		Task runningTask = engine.getTask(taskId);
        TaskStepInfo step;
        if (runningTask != null) {
		    step = runningTask.getStep(stepNr);
        } else {
            DeploymentTaskInfo archivedTask = taskArchive.getTask(taskId);
            step = archivedTask.getStep(stepNr);
        }

        if (hasBeenModifiedSince(step, ifModifiedSince)) {
			return ResponseFactory.ok(dtoConverter.taskStepInfoToDto(stepNr, step)).lastModified(step.getLastModificationDate().getTime()).build();
		} else {
			return ResponseFactory.notModified().build();
		}
	}

	protected boolean hasBeenModifiedSince(TaskStepInfo step, Date ifModifiedSince) {
		if (ifModifiedSince == null) {
			return true;
		}

		Calendar lastModifiedCal = step.getLastModificationDate();
		lastModifiedCal.set(Calendar.MILLISECOND, 0); // set to 0 because time resolution is to 1 second in HTTP format
		return ifModifiedSince.getCalendar().before(lastModifiedCal);
	}


	@Override
	public Response getUnfinishedTasks() {
		return ResponseFactory.ok(dtoConverter.tasksToDto(engine.getAllIncompleteTasksForCurrentUser())).build();
	}


	@Override
    public Response getStepsForTask(String taskId) {
		Task runningTask = engine.getTask(taskId);
        Steps steps;
        if (runningTask != null) {
            steps = dtoConverter.taskStepInfosToDto(runningTask.getSteps());
        } else {
            steps = dtoConverter.taskStepInfosToDto(taskArchive.getTask(taskId).getSteps());
        }
		steps.setTaskId(taskId);
		return ResponseFactory.ok(steps).build();
    }

	private static final Logger logger = LoggerFactory.getLogger(TaskResource.class);
}
