package com.xebialabs.deployit.core.rest.api.reports;

import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.collect.MapMaker;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.DeploymentTaskInfo;
import com.xebialabs.deployit.task.TaskArchive;
import com.xebialabs.deployit.task.jcrarchive.JcrTaskArchive.TaskCallback;

public class AggregatedDeploymentsKeyIndicator {
	private TaskArchive taskArchive;

	public AggregatedDeploymentsKeyIndicator(TaskArchive taskArchive) {
		this.taskArchive = taskArchive;
	}

	@SuppressWarnings("deprecation")
    public Report report(final Date startDate, final Date endDate, final ReportFilterType filterType, List<String> filterCriteria) {
		final Map<String, TaskStateData> tasksMap = new MapMaker().makeComputingMap(new Function<String, TaskStateData>() {
			public TaskStateData apply(String input) {
				return new TaskStateData();
			}
		});

		ArchivedTaskSearchParameters searchTasks = new ArchivedTaskSearchParameters();
		searchTasks.createdBetween(startDate.getCalendar(), endDate.getCalendar());

		for (String appOrEnvName : filterCriteria) {
			switch (filterType) {
			case APPLICATION:
				searchTasks.forApplication(appOrEnvName);
				break;
			case ENVIRONMENT:
				searchTasks.toEnvironment(appOrEnvName);
				break;
			default:
				throw new UnsupportedOperationException("Report filter type, " + filterType + " is not supported");
			}
			taskArchive.searchTasksWithoutLoadingSteps(searchTasks, new TaskCallback() {
				public void doWithTask(DeploymentTaskInfo task) {
					fillTasksMap(tasksMap, task, filterType);
				}
			});
		}

		return generateReportDataFromTaskMap(tasksMap);
	}

	private Report generateReportDataFromTaskMap(Map<String, TaskStateData> tasksMap) {
		final Report report = new Report();
		for (Map.Entry<String, TaskStateData> entry : tasksMap.entrySet()) {
			final Report.ReportLine reportLine = report.addLine();
			reportLine.addValue("entityName", entry.getKey());
			reportLine.addValue("noOfSuccessfulDeployments", entry.getValue().getNoOfSuccessfulDeployments().toString());
			reportLine.addValue("noOfFailedDeployments", entry.getValue().getNoOfFailedDeployments().toString());
			reportLine.addValue("noOfAbortedDeployments", entry.getValue().getNoOfAbortedDeployments().toString());
			String averageDeploymentTime = "00:00:00";

			if (entry.getValue().getTotalSuccessfulTasks() > 0) {
				// Create a DateFormatter object for displaying date in specified format.
				// Create a calendar object that will convert the date and time value in milliseconds to date.
				averageDeploymentTime = ReportUtils.formatToHoursMinsSecs(entry.getValue().getAverageDeploymentTime() / entry.getValue().getTotalSuccessfulTasks());
			}
			reportLine.addValue("averageDuration", averageDeploymentTime);
		}
		return report;
	}

	private void fillTasksMap(Map<String, TaskStateData> tasksMap, DeploymentTaskInfo task, final ReportFilterType filterType) {
		String entityId = getEntityId(task, filterType);
		TaskStateData taskStateData = tasksMap.get(entityId);
		taskStateData.addTaskAndDuration(task);
	}

	private String getEntityId(DeploymentTaskInfo task, ReportFilterType filterType) {
		switch (filterType) {
		case APPLICATION:
			return task.getApplicationName();
		case ENVIRONMENT:
			return task.getEnvironment();
		}
		throw new IllegalArgumentException("Unknown filter type: " + filterType);
	}

}
