package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.RepositoryProxy;
import com.xebialabs.deployit.core.api.dto.ArtifactAndData;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDtos;
import com.xebialabs.deployit.core.api.dto.RepositoryObject;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.validation.Validator;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.ws.rs.core.Response;
import java.util.Collection;
import java.util.List;

import static javax.ws.rs.core.Response.Status;
import static javax.ws.rs.core.Response.Status.BAD_REQUEST;

@Controller
public class RepositoryResource extends AbstractSecuredResource implements RepositoryProxy {

    @Autowired
    private RepositoryService repositoryService;

	@Autowired
	private ConfigurationItemDtoReader dtoReader;

	@Autowired
	private ConfigurationItemDtoWriter dtoWriter;

    @Autowired
    private Validator validator;

	@Autowired
	private WorkDirFactory workDirFactory;

    @Override
    public Response create(String id, ConfigurationItemDto dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.setId(id);
	    ConfigurationItem ci = dtoReader.read(dto);
        return createInternal(ci);
    }

    @Override
    public Response create(String id, ArtifactAndData dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.getArtifact().setId(id);
	    WorkDir workDir = workDirFactory.newWorkDir();
	    try {
			ConfigurationItem ci = dtoReader.read(dto, workDir);
			return createInternal(ci);
	    } finally {
		    workDir.delete();
	    }
    }

    @Override
    public Response createMultiple(ConfigurationItemDtos dto) {
	    checkPermission(Permission.EDIT_REPO, Lists.transform(dto.getObjects(), new Function<RepositoryObject, String>() {
		    public String apply(RepositoryObject from) {
			    return from.getId();
		    }
	    }));
        return createInternal(dtoReader.read(dto));
    }

    @Override
    public Response read(String id) {
	    ConfigurationItem entity = repositoryService.read(id);
        final RepositoryObject dto = dtoWriter.write(entity);
        return ResponseFactory.status(Status.OK).entity(dto).build();
    }

    @Override
    public Response update(String id, ConfigurationItemDto dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.setId(id);
	    ConfigurationItem ci = dtoReader.read(dto);
        return updateInternal(ci);
    }

    @Override
    public Response update(String id, ArtifactAndData dto) {
	    checkPermission(Permission.EDIT_REPO, id);
        dto.getArtifact().setId(id);
	    WorkDir workDir = workDirFactory.newWorkDir();
	    try {
			ConfigurationItem ci = dtoReader.read(dto, workDir);
			return updateInternal(ci);
	    } finally {
		    workDir.delete();
	    }
    }

    @Override
    public Response delete(String id) {
	    checkPermission(Permission.EDIT_REPO, id);
        repositoryService.delete(id);
        return ResponseFactory.noContent().build();
    }

    private Response createInternal(final ConfigurationItem ci) {
        Response response = validateEntityAndCreateErrorResponseIfNeeded(ci);
        if (response != null) {
            return response;
        }

        repositoryService.create(ci);
        return reloadEntityAndCreateSuccessResponse(ci);
    }

	private Response validateEntityAndCreateErrorResponseIfNeeded(ConfigurationItem ci) {
		List<ValidationMessage> messages = validator.validate(ci, Lists.<ConfigurationItem>newArrayList());
		if (!messages.isEmpty()) {
		    return ResponseFactory.status(BAD_REQUEST).entity(dtoWriter.write(ci, messages)).build();
		}
		return null;
	}

    private Response validateEntityAndCreateErrorResponseIfNeeded(Collection<ConfigurationItem> entities) {
		ConfigurationItemDtos repositoryObjects = new ConfigurationItemDtos();
		boolean allValid = true;
	    ConfigurationItem[] entitiesArray = entities.toArray(new ConfigurationItem[entities.size()]);
		for (ConfigurationItem entity : entities) {
			List<ValidationMessage> messages = validator.validate(entity, entitiesArray);
			repositoryObjects.add(dtoWriter.write(entity, messages));
			allValid = allValid && messages.isEmpty();
		}
		if (!allValid) {
			return ResponseFactory.status(BAD_REQUEST).entity(repositoryObjects).build();
		}
        return null;
    }

    private Response reloadEntityAndCreateSuccessResponse(ConfigurationItem ci) {
        ConfigurationItem reloaded = repositoryService.read(ci.getId());
        return ResponseFactory.created(dtoWriter.write(reloaded)).build();
    }

    private Response createInternal(Collection<ConfigurationItem> cis) {
        Response response = validateEntityAndCreateErrorResponseIfNeeded(cis);
        if (response != null) {
            return response;
        }

	    ConfigurationItem[] cisArray = cis.toArray(new ConfigurationItem[cis.size()]);
        repositoryService.create(cisArray);

        return reloadEntityAndCreateSuccessResponse(cis);
    }

    private Response reloadEntityAndCreateSuccessResponse(Collection<ConfigurationItem> cis) {
        List<ConfigurationItem> reloadedDtos = Lists.newArrayList();

        for (ConfigurationItem entity : cis) {
            ConfigurationItem reloadedEntity = repositoryService.read(entity.getId());
            reloadedDtos.add(reloadedEntity);
        }

        return ResponseFactory.created(dtoWriter.write(reloadedDtos)).build();
    }

    private Response updateInternal(final ConfigurationItem ci) {
        Response response = validateEntityAndCreateErrorResponseIfNeeded(ci);
        if (response != null) {
            return response;
        }

        repositoryService.update(ci);
        return reloadEntityAndCreateSuccessResponse(ci);
    }
}
