package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Function;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.core.api.dto.*;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.utils.Strings;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.permission.Permission;
import org.springframework.stereotype.Component;

import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.transform;
import static java.lang.String.format;

@Component
public class DtoReader {
	
	public static final Function<ConfigurationItemData, ConfigurationItemId> ciDataToCiId = new Function<ConfigurationItemData, ConfigurationItemId>() {
		public ConfigurationItemId apply(ConfigurationItemData input) {
			return new ConfigurationItemId(input.getId(), input.getType().toString());
		}
	};

	public static final Function<ConfigurationItemId, ConfigurationItemData> ciIdToCiData = new Function<ConfigurationItemId, ConfigurationItemData>() {
		public ConfigurationItemData apply(ConfigurationItemId input) {
			Type type = Strings.isBlank(input.getType()) ? null : Type.valueOf(input.getType());
			return new ConfigurationItemData(input.getId(), type);
		}
	};
	
	public List<Role> readRoleAssignments(RoleAssignments assignments) {
		List<Role> roles = newArrayList();
		for (RoleAssignment roleAssignment : assignments.getAssignments()) {
			com.xebialabs.deployit.core.api.dto.Role roleDto = roleAssignment.getRole();
			Role role = readRole(roleDto);
			List<String> principals = roleAssignment.getPrincipals();
			role.getPrincipalsAssigned().addAll(principals);
			roles.add(role);
		}

		return roles;
	}

	private Role readRole(com.xebialabs.deployit.core.api.dto.Role roleDto) {
		return new Role(roleDto.getId(), roleDto.getName());
	}


	public Multimap<Role, Permission> readRolePermissions(RolePermissions permissions, List<Role> roles) {
		Multimap<Role, Permission> multimap = HashMultimap.create();
		for (RolePermission rolePermission : permissions.getRolePermissions()) {
			Role role = readRole(rolePermission.getRole());
			if (role.getId() == null) {
				role = findRole(role.getName(), roles);
			}
			List<Permission> permissionList = transform(rolePermission.getPermissions(), new Function<String, Permission>() {
				public Permission apply(String input) {
					return Permission.find(input);
				}
			});
			multimap.putAll(role, permissionList);
		}
		return multimap;
	}

	private Role findRole(String name, List<Role> roles) {
		for (Role role : roles) {
			if (role.getName().equals(name)) {
				return role;
			}
		}
		throw new IllegalArgumentException(format("Couldn't find role with name [%s]", name));
	}
}
