package com.xebialabs.deployit.core.rest.api;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.core.api.SecurityProxy;
import com.xebialabs.deployit.core.api.dto.*;
import com.xebialabs.deployit.core.api.resteasy.http.tunnel.ResponseFactory;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.exception.DeployitException;
import com.xebialabs.deployit.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.PermissionLister;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.authentication.RememberMe;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Controller;

import javax.ws.rs.PathParam;
import javax.ws.rs.core.Response;
import java.util.List;


@Controller
public class SecurityResource extends AbstractSecuredResource implements SecurityProxy {

	@Autowired
	private RoleService roleService;

	@Autowired
	private PermissionEditor permissionEditor;

	@Autowired
	private PermissionLister permissionLister;

	@Autowired
	private DtoWriter dtoWriter;

	@Autowired
	private DtoReader dtoReader;

    @Override
    public Response logout() {
        RememberMe.forgetMe();
        return ResponseFactory.ok().build();
    }

	@Override
	public Response getGrantedPermissions(String id) {
		checkPermission(Permission.EDIT_SECURITY);
		if (id.equals("global")) {
			logger.debug("Requested global permissions.");
			id = "";
		}

		Multimap<Role, Permission> rolePermissionMultimap = permissionEditor.readPermissions(id);
		List<RolePermission> rolePermissions = dtoWriter.writeRolePermissions(rolePermissionMultimap);
		return ResponseFactory.ok(new RolePermissions(rolePermissions)).build();
	}

	@Override
	public Response writeGrantedPermissions(String id, RolePermissions permissions) {
		checkPermission(Permission.EDIT_SECURITY);
		if (id.equals("global")) {
			logger.debug("Writing global permissions");
			id = "";
		}

		List<Role> roles = roleService.readRoles();
		Multimap<Role, Permission> multimap = dtoReader.readRolePermissions(permissions, roles);
		permissionEditor.editPermissions(id, multimap);
		return ResponseFactory.ok().build();
	}

	@Override
	public Response getRoleAssignments() {
		checkPermission(Permission.EDIT_SECURITY);
		List<Role> multimap = roleService.readRoleAssignments();
		logger.debug("RoleService delivered roles: {}", multimap);
		List<RoleAssignment> assignments = dtoWriter.writeRoleAssignments(multimap);
		logger.debug("Going to return the following roles: {}", assignments);
		return ResponseFactory.ok(new RoleAssignments(assignments)).build();
	}

	@Override
	public Response writeRoleAssignments(RoleAssignments assignments) {
		checkPermission(Permission.EDIT_SECURITY);
		logger.debug("Got the following roles: {}", assignments);
		List<Role> roles = dtoReader.readRoleAssignments(assignments);
		logger.debug("Going to write the following role assignments: {}", roles);
		roleService.writeRoleAssignments(roles);
		return ResponseFactory.ok().build();
	}

	@Override
	public Response getRoles() {
		List<Role> roles = roleService.readRoles();
		return ResponseFactory.ok(new Roles(dtoWriter.writeRoles(roles))).build();
	}

	@Override
	public Response hasPermission(String permission, String id) {
		checkPermission(Permission.find(permission), id);
		return ResponseFactory.ok().build();
	}
	
	@Override
    public Response retrievePermissions(final String roleName) {
		Role role = roleService.getRoleForRoleName(roleName);
		Multimap<String, String> grantedPermissions = permissionLister.listPermissions(role);
        PrincipalPermissions entity = new PrincipalPermissions(roleName, grantedPermissions);
        return ResponseFactory.ok(entity).build();
    }

	@Override
	public Response retrievePermissionsForCurrentUser() {
		Authentication authentication = Permissions.getAuthentication();
		List<Role> roles = roleService.getRolesFor(authentication);
		Multimap<String, String> grantedPermissions = permissionLister.listPermissions(roles);
		PrincipalPermissions entity = new PrincipalPermissions(authentication.getName(), grantedPermissions);
		return ResponseFactory.ok(entity).build();
	}
	
    private Permission getPermission(final String permissionName) {
        final Permission permission = Permission.find(permissionName);
        if (permission == null) {
            throw new UnknownPermissionException(permissionName);
        }
        return permission;
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class UnknownPermissionException extends DeployitException {
        public UnknownPermissionException(String permission) {
            super("Permission %s does not exist.", permission);
        }
    }
	private static final Logger logger = LoggerFactory.getLogger(SecurityResource.class);
}




