package com.xebialabs.deployit.core.rest.api;

import java.util.List;
import javax.ws.rs.PathParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.InspectionService;
import com.xebialabs.deployit.engine.api.dto.Inspection;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.tasker.Engine;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.service.discovery.DiscoveryResult;
import com.xebialabs.deployit.service.discovery.DiscoveryService;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;

/**
 */
@Controller
public class DiscoveryResource extends AbstractSecuredResource implements InspectionService {

    @Autowired
    private DiscoveryService discoveryService;

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private Engine engine;

    @Autowired
    private JcrTaskArchive taskArchive;

    @Override
    public Inspection prepare(ConfigurationItem configurationItem) {
        checkPermission(Permission.DISCOVERY);
        return new Inspection(configurationItem);
    }

    @Override
    public Inspection prepare(String id) {
        checkPermission(Permission.DISCOVERY);
        ConfigurationItem read = repositoryService.read(id);
        return new Inspection(read);
    }

    @Override
    @Workdir
    public String createTask(Inspection inspection) {
        checkPermission(Permission.DISCOVERY);
        try {
            TaskSpecification taskSpecification = discoveryService.prepareDiscovery(inspection.getConfigurationItem(), WorkdirHolder.get());
            logger.info("Creating inspection task for [{}]", inspection.getConfigurationItem());
            return engine.register(taskSpecification);
        } catch (RuntimeException e) {
            WorkdirHolder.get().delete();
            throw e;
        }
    }

    @Override
    public List<ConfigurationItem> retrieveInspectionResults(@PathParam("taskId") String taskId) {
        checkPermission(Permission.DISCOVERY);
        TaskState task = taskArchive.getTask(taskId);
        if (task == null) {
            throw new Checks.ConditionFailedException("Task [%s] is not yet finished.", taskId);
        }
        DiscoveryResult discoveryResults = discoveryService.getDiscoveryResults(taskId);
        if (!discoveryResults.getOwner().equals(Permissions.getAuthenticatedUserName())) {
            throw new Checks.ConditionFailedException("You were not the owner of task [%s]", taskId);
        }
        return discoveryResults.getItems();
    }

    private static final Logger logger = LoggerFactory.getLogger(DiscoveryResource.class);
}
