package com.xebialabs.deployit.core.rest.api;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;

import static com.google.common.collect.Iterables.filter;

public class ExposedDescriptor implements Descriptor {

    private final static Predicate<PropertyDescriptor> EXPOSED_PROPERTIES = new Predicate<PropertyDescriptor>() {
        @Override
        public boolean apply(PropertyDescriptor input) {
            return !input.isHidden();
        }
    };

    private final Descriptor descriptor;
    private final boolean exposeHiddenProperties;

    public ExposedDescriptor(Descriptor descriptor) {
        this(descriptor, true);
    }

    public ExposedDescriptor(Descriptor descriptor, boolean exposeHiddenProperties) {
        this.descriptor = descriptor;
        this.exposeHiddenProperties = exposeHiddenProperties;
    }

    @Override
    public Type getType() {
        return descriptor.getType();
    }

    @Override
    public Class<?> getClazz() {
        return descriptor.getClazz();
    }

    @Override
    public String getDescription() {
        return descriptor.getDescription();
    }

    @Override
    public ConfigurationItemRoot getRoot() {
        return descriptor.getRoot();
    }

    @Override
    public Collection<PropertyDescriptor> getPropertyDescriptors() {
        if (!exposeHiddenProperties) {
            return toCollection(filter(descriptor.getPropertyDescriptors(), EXPOSED_PROPERTIES));
        }
        return descriptor.getPropertyDescriptors();
    }

    private static <E> Collection<E> toCollection(Iterable<E> iterable) {
        return (iterable instanceof Collection)
            ? (Collection<E>) iterable
            : Lists.newArrayList(iterable.iterator());
    }

    @Override
    public PropertyDescriptor getPropertyDescriptor(String name) {
        PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(name);
        if (propertyDescriptor != null && propertyDescriptor.isHidden() && !exposeHiddenProperties) {
            return null;
        }
        return propertyDescriptor;
    }

    @Override
    public MethodDescriptor getControlTask(String name) {
        return descriptor.getControlTask(name);
    }

    @Override
    public Collection<MethodDescriptor> getControlTasks() {
        return descriptor.getControlTasks();
    }

    @Override
    public boolean isAssignableTo(Class<?> clazz) {
        return descriptor.isAssignableTo(clazz);
    }

    @Override
    public boolean isAssignableTo(Type type) {
        return descriptor.isAssignableTo(type);
    }

    @Override
    public List<Type> getSuperClasses() {
        return descriptor.getSuperClasses();
    }

    @Override
    public Set<Type> getInterfaces() {
        return descriptor.getInterfaces();
    }

    @Override
    public boolean isVirtual() {
        return descriptor.isVirtual();
    }

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return descriptor.areEqual(item, other);
    }

    @Override
    public <T extends ConfigurationItem> T newInstance() {
        return descriptor.<T>newInstance();
    }

    @Override
    public Type getDeployableType() {
        return descriptor.getDeployableType();
    }

    @Override
    public Type getContainerType() {
        return descriptor.getContainerType();
    }

    @Override
    public List<ValidationMessage> validate(ConfigurationItem ci) {
        return descriptor.validate(ci);
    }

    @Override
    public boolean isInspectable() {
        return descriptor.isInspectable();
    }

    @Override
    public String toString() {
        return descriptor.toString();
    }
}
