package com.xebialabs.deployit.core.rest.api;

import java.io.File;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.PackageService;
import com.xebialabs.deployit.engine.api.dto.FileUpload;
import com.xebialabs.deployit.engine.spi.event.PackageImportedEvent;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.server.api.importer.ImportSource;
import com.xebialabs.deployit.service.importer.ImporterException;
import com.xebialabs.deployit.service.importer.ImporterService;
import com.xebialabs.deployit.service.importer.source.FileSource;
import com.xebialabs.deployit.service.importer.source.InputStreamSource;
import com.xebialabs.deployit.service.importer.source.UrlSource;

import static java.lang.String.format;


/**
 * I am A REST resource that exposes functionality on the Deployable packages.
 * <p/>
 * Functionality includes: - Listing all Deployment packages in the importablePackages directory on the server.
 */
@Controller
public class PackageResource extends AbstractSecuredResource implements PackageService {

    @Autowired
    private ImporterService importerService;

    @Autowired
    private RepositoryService repositoryService;

    @Override
    public List<String> list() {
        List<String> files = importerService.listPackages();
        return files;
    }

    public ConfigurationItem importPackage(String file) {
        if (!importerService.getImportablePackageDirectory().exists()) {
            throw new ImporterException(format("Directory \'%s\' to import packages from does not exist", importerService.getImportablePackageDirectory().getAbsolutePath()));
        }
        FileSource fileSource = new FileSource(new File(importerService.getImportablePackageDirectory(), file), false);
        return doImport(fileSource);
    }

    @Override
    public ConfigurationItem fetch(String url) {
        try {
            UrlSource urlSource = new UrlSource(new URL(url));
            return doImport(urlSource);
        } catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public ConfigurationItem upload(String file, FileUpload form) {
        InputStream uploadedPackage = form.getFileData();
        InputStreamSource byteArraySource = new InputStreamSource(file, uploadedPackage);
        return doImport(byteArraySource);
    }

    private ConfigurationItem doImport(ImportSource source) {
        String importedApplication = importerService.importPackage(source);
        ConfigurationItem read = repositoryService.read(importedApplication);

        EventBusHolder.publish(new PackageImportedEvent(importedApplication));
        
        return read;
    }
}
