package com.xebialabs.deployit.core.rest.api;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.google.common.collect.Multimap;

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.PermissionService;
import com.xebialabs.deployit.engine.spi.event.PermissionGrantedEvent;
import com.xebialabs.deployit.engine.spi.event.PermissionRevokedEvent;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.PermissionLister;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;

import static com.xebialabs.deployit.security.permission.Permission.EDIT_SECURITY;
import static com.xebialabs.deployit.security.permission.Permission.find;

@Controller
public class PermissionResource extends AbstractSecuredResource implements PermissionService {

    @Autowired
    private RoleService roleService;

    @Autowired
    private PermissionEditor permissionEditor;

    @Autowired
    private PermissionLister permissionLister;

    private static Permission getPermission(final String permissionName) {
        final Permission permission = find(permissionName);
        if (permission == null) {
            throw new UnknownPermissionException(permissionName);
        }
        return permission;
    }

    @Override
    public boolean isGranted(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        Multimap<String, String> stringStringMultimap = permissionLister.listPermissions(role);
        return stringStringMultimap.containsEntry(id, permission);
    }

    @Override
    public void grant(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        Multimap<Role, Permission> rolePermissions = permissionEditor.readPermissions(id);
        rolePermissions.put(role, getPermission(permission));
        permissionEditor.editPermissions(id, rolePermissions);
        
        EventBusHolder.publish(new PermissionGrantedEvent(id, roleName, permission));
    }

    @Override
    public void revoke(String permission, String id, String roleName) {
        checkPermission(EDIT_SECURITY);
        Role role = roleService.getRoleForRoleName(roleName);
        Multimap<Role, Permission> rolePermissions = permissionEditor.readPermissions(id);
        rolePermissions.remove(role, getPermission(permission));
        permissionEditor.editPermissions(id, rolePermissions);

        EventBusHolder.publish(new PermissionRevokedEvent(id, roleName, permission));
    }

    @Override
    public void checkMyPermission(String permission, String id) {
        checkPermission(getPermission(permission), id);
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class UnknownPermissionException extends DeployitException {
        public UnknownPermissionException(String permission) {
            super("Permission %s does not exist.", permission);
        }
    }
}