package com.xebialabs.deployit.core.rest.api;

import java.util.Collection;
import java.util.List;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.TaskService;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.engine.api.execution.TaskWithSteps;
import com.xebialabs.deployit.engine.spi.event.*;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.engine.tasker.Engine;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.engine.tasker.TaskNotFoundException;
import com.xebialabs.deployit.engine.tasker.TaskStep;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static java.lang.String.format;

@Controller
public class TaskResource extends AbstractSecuredResource implements TaskService {

    @Autowired
    private Engine engine;

    @Autowired
    private JcrTaskArchive taskArchive;

    @Override
    public void start(final String taskId) {
        engine.execute(taskId);

        EventBusHolder.publish(new TaskStartedEvent(taskId));
    }

    @Override
    public void cancel(final String taskId) {
        engine.cancel(taskId);

        EventBusHolder.publish(new TaskCancelledEvent(taskId));
    }

    @Override
    public void archive(final String taskId) {
        engine.archive(taskId);
        EventBusHolder.publish(new TaskArchivedEvent(taskId));
    }

    @Override
    public void stop(final String taskId) {
        try {
            engine.stop(taskId);
            EventBusHolder.publish(new TaskStoppedEvent(taskId));
        } catch (IllegalStateException ise) {
            logger.error(format("Could not stop task %s", taskId), ise);
            throw ise;
        }
    }

    @Override
    public void abort(final String taskId) {
        try {
            engine.abort(taskId);
            EventBusHolder.publish(new TaskAbortedEvent(taskId));
        } catch (IllegalStateException ise) {
            logger.error(format("Could not abort task %s", taskId), ise);
            throw ise;
        }
    }

    @Override
    public TaskState getTask(final String taskId) {
        try {
            return new TaskStateView(engine.retrieve(taskId));
        } catch (TaskNotFoundException tnfe) {
            return new TaskStateView(taskArchive.getTask(taskId));
        }
    }

    @Override
    public TaskState assign(final String taskId, final String owner) {
        // Rules:
        // - admin can reassign everyone's tasks
        // - user can reassign his own tasks
        final com.xebialabs.deployit.engine.tasker.Task task = engine.retrieve(taskId);
        checkArgument(task != null, "Could not find active task %s", taskId);

        Task retrieve = engine.retrieve(taskId);
        if (hasPermission(Permission.ADMIN)) {
            retrieve.setOwner(owner);
        } else if (hasPermission(Permission.TASK_ASSIGN) && retrieve.getOwner().equals(Permissions.getAuthenticatedUserName())) {
            retrieve.setOwner(owner);
        } else {
            throw PermissionDeniedException.withMessage("Neither ADMIN nor TASK_ASSIGN permissions are granted to you or you are not the owner of the task");
        }

        EventBusHolder.publish(new TaskAssignedEvent(taskId, owner));

        return new TaskStateView(task);
    }

    @Override
    public TaskWithSteps skip(String taskId, List<Integer> stepIds) {
        return skip(taskId, stepIds, true);
    }

    @Override
    public TaskWithSteps unskip(String taskId, List<Integer> stepIds) {
        return skip(taskId, stepIds, false);
    }

    private TaskWithSteps skip(String taskId, List<Integer> stepIds, boolean skip) {

        final com.xebialabs.deployit.engine.tasker.Task task = engine.retrieve(taskId);

        checkArgument(task != null, "Could not find active task %s", taskId);
        checkPermission(Permission.TASK_SKIPSTEP, task);

        if (skip) {
            engine.skipSteps(taskId, stepIds);
        } else {
            engine.unskipSteps(taskId, stepIds);
        }
        return task;
    }

    private void checkPermission(Permission permission, com.xebialabs.deployit.engine.tasker.Task task) {
        if (task.getMetadata().containsKey("environment_id")) {
            String env = task.getMetadata().get("environment_id");
            checkPermission(permission, env);
        } else {
            checkPermission(permission);
        }
    }

    @Override
    public TaskWithSteps moveStep(String taskId, int stepNr, int newPosition) {
        final com.xebialabs.deployit.engine.tasker.Task task = engine.retrieve(taskId);
        checkArgument(task != null, "Could not find active task %s", taskId);
        checkPermission(Permission.TASK_MOVESTEP, task);

        engine.moveStep(taskId, stepNr, newPosition);

        return task;
    }

    @Override
    public TaskWithSteps addPause(final String taskId, final int stepNr) {
        engine.addPauseStep(taskId, stepNr);
        return engine.retrieve(taskId);
    }

    @Override
    public StepState getStep(String taskId, int stepNr, DateTime ifModifiedSince) {
        StepState step;
        try {
            com.xebialabs.deployit.engine.tasker.Task runningTask = engine.retrieve(taskId);
            step = runningTask.getStep(stepNr);
        } catch (TaskNotFoundException tnfe) {
            TaskWithSteps archivedTask = taskArchive.getTask(taskId);
            step = archivedTask.getStep(stepNr);
        }

        if (hasBeenModifiedSince(step, ifModifiedSince)) {
            return step;
        } else {
            throw new NotModifiedException();
        }
    }

    protected boolean hasBeenModifiedSince(StepState step, DateTime ifModifiedSince) {
        if (ifModifiedSince == null) {
            return true;
        }

        // Check either completionDate (when step is finished, or LMD when step is still in active task.
        DateTime lastModifiedCal = step.getCompletionDate();
        if (step instanceof TaskStep) {
            lastModifiedCal = ((TaskStep) step).getLastModificationDate();
        }
        // Set milliseconds to 0 because time resolution is to 1 second in HTTP format
        return lastModifiedCal.withMillis(0).isAfter(ifModifiedSince);
    }

    /**
     * Lists all unfinished tasks for the current user.
     * <p/>
     * NOTE: this method is invoked from the GUI when restarting the UI after a
     * crash. Therefore, it should only return the tasks for the currently
     * logged in user to prevent logging in as an administrator and seeing all
     * tasks in the system open in your GUI.
     * <p/>
     * If you do need the latter functionality, see getAllUnfinishedTasks().
     */
    @Override
    public List<TaskState> getMyCurrentTasks() {
        final String principal = Permissions.getAuthenticatedUserName();
        return Lists.<TaskState>newArrayList(getAllTasks().filter(new Predicate<TaskState>() {
            @Override
            public boolean apply(TaskState input) {
                return input.getOwner().equals(principal);
            }
        }));
    }

    private FluentIterable<TaskStateView> getAllTasks() {
        return FluentIterable.from(engine.getAllIncompleteTasks()).transform(new Function<Task, TaskStateView>() {
            @Override
            public TaskStateView apply(Task input) {
                return new TaskStateView(input);
            }
        });
    }

    /**
     * Lists all unfinished tasks in the system.
     */
    @Override
    public List<TaskState> getAllCurrentTasks() {
        return Lists.<TaskState>newArrayList(getAllTasks());
    }

    @Override
    public TaskWithSteps getSteps(String taskId) {
        try {
            return engine.retrieve(taskId);
        } catch (TaskNotFoundException tnfe) {
            return taskArchive.getTask(taskId);
        }
    }

    @Override
    public List<TaskState> query(LocalDate begin, LocalDate end) {
        return newArrayList(FluentIterable.from(search(begin, end)).transform(new Function<TaskWithSteps, TaskState>() {
            @Override
            public TaskState apply(TaskWithSteps input) {
                return new TaskStateView(input);
            }
        }));
    }

    @Override
    public List<TaskWithSteps> export(LocalDate begin, LocalDate end) {
        return Lists.<TaskWithSteps>newArrayList(search(begin, end));
    }

    private Collection<ArchivedTask> search(LocalDate begin, LocalDate end) {

        ArchivedTaskSearchParameters parameters = new ArchivedTaskSearchParameters();
        if (end == null) {
            end = new LocalDate();
        }
        if (begin != null) {
            parameters.createdBetween(begin.toDateTimeAtStartOfDay(), end.toDateTimeAtStartOfDay());
        }

        return taskArchive.searchTasks(parameters);
    }

    private static final Logger logger = LoggerFactory.getLogger(TaskResource.class);

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 304)
    private static class NotModifiedException extends DeployitException {
    }
}
