package com.xebialabs.deployit.core.rest.api.reports;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;

import org.joda.time.Duration;
import org.joda.time.Period;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;

import com.google.common.base.Function;
import com.google.common.base.Joiner;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.engine.api.execution.TaskState;

import static com.google.common.collect.FluentIterable.from;
import static com.google.common.collect.Lists.newArrayList;

public class ReportUtils {
    public static final DateTimeFormatter FORMATTER = DateTimeFormat.forPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZ");

    static final PeriodFormatter HH_MM_SS = new PeriodFormatterBuilder().minimumPrintedDigits(2).printZeroAlways().appendHours().appendSeparator(":").appendMinutes().appendSeparator(":").appendSeconds().toFormatter();
    static final PeriodFormatter MM_SS = new PeriodFormatterBuilder().printZeroAlways().minimumPrintedDigits(1).appendMinutes().appendSeparator(":").appendSeconds().toFormatter();

    private static final Joiner LINE_JOINER = Joiner.on("\n");
    private static final Joiner FIELD_JOINER = Joiner.on(",");
    private static final Function<Object, String> QUOTE = new Function<Object, String>() {
        public String apply(final Object input) {
            return String.format("\"%s\"", input);
        }
    };

    static final String formatToMinsAndSecs(long millis) {
        Period period = new Duration(millis).toPeriod();
        return MM_SS.print(period);
    }

    static final String formatToHoursMinsSecs(long millis) {
        Period period = new Duration(millis).toPeriod();
        return HH_MM_SS.print(period);
    }

    public static long duration(TaskState s) {
        return s.getCompletionDate().getMillis() - s.getStartDate().getMillis();
    }

    public static String toCsv(Report report) {
        List<Report.ReportLine> lines = report.getLines();
        List<String> csvLines = newArrayList();
        if (lines.size() > 0) {
            addLine(csvLines, sortedHeaders(lines));
            for (Report.ReportLine line : lines) {
                addLine(csvLines, sortedValues(line));
            }
        }
        return LINE_JOINER.join(csvLines);
    }

    private static void addLine(List<String> csvLines, Iterable<?> fields) {
        csvLines.add(FIELD_JOINER.join(from(fields).transform(QUOTE)));
    }

    private static Set<String> sortedHeaders(List<Report.ReportLine> lines) {
        return new TreeMap<String, Object>(lines.iterator().next().getValues()).keySet();
    }

    private static Collection<Object> sortedValues(Report.ReportLine line) {
        return new TreeMap<String, Object>(line.getValues()).values();
    }
}
