package com.xebialabs.deployit.core.rest.api.reports.widgets;

import java.util.*;
import java.util.concurrent.TimeUnit;

import org.joda.time.DateTime;

import com.google.common.base.Function;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.engine.api.execution.TaskState;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;
import com.xebialabs.deployit.task.archive.JcrTaskArchive.TaskCallback;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.core.rest.api.reports.ReportUtils.duration;
import static com.xebialabs.deployit.task.TaskType.INITIAL;
import static com.xebialabs.deployit.task.TaskType.UNDEPLOY;
import static com.xebialabs.deployit.task.TaskType.UPGRADE;

/*********************
 * Deployment duration over time widget
 */
public class DeploymentTrendsPercentileWidget extends DashboardWidgetBase {

    private static final int EIGHTIETH_PERCENTILE = 80;

    private static enum ReportTimeScale {
        SECS, MINS, HRS;
    }

    public DeploymentTrendsPercentileWidget(final JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public Report getReport(final DateTime begin, final DateTime end) {
        Map<String, List<TaskState>> groupedTasks = groupSuccesfulTasksByMonth(begin, end);
        Map<String, Long> monthPercentileMap = calculatePercentiles(groupedTasks);
        return generateReportData(monthPercentileMap);
    }

    private Map<String, List<TaskState>> groupSuccesfulTasksByMonth(DateTime begin, DateTime end) {
        ArchivedTaskSearchParameters params = getSearchParameters().createdBetween(begin, end).thatCompleted().thatAreOfType(EnumSet.of(INITIAL, UNDEPLOY, UPGRADE));
        final Map<String, List<TaskState>> groupedTasks = initializeMap(begin, end, new Function<String, List<TaskState>>() {
            public List<TaskState> apply(String input) {
                return newArrayList();
            }
        });

        taskArchive.searchTasksWithoutLoadingSteps(params, new TaskCallback() {
            public void doWithTask(ArchivedTask task) {
                String month = monthFormat.print(task.getCompletionDate());
                List<TaskState> dataPerMonth = groupedTasks.get(month);
                if(dataPerMonth != null) {
                    dataPerMonth.add(task);
                }
            }
        });

        return groupedTasks;
    }

    private static Map<String, Long> calculatePercentiles(Map<String, List<TaskState>> groupedTasks) {
        return Maps.transformValues(groupedTasks, new Function<List<TaskState>, Long>() {
            @Override
            public Long apply(List<TaskState> input) {
                return getPercentile(input, EIGHTIETH_PERCENTILE);
            }
        });
    }

    /**
     * Calculates percentile by given ratio by using nearest rank algorithm.
     *
     * @link http://en.wikipedia.org/wiki/Percentile
     * @param tasks
     * @param percentileRatio
     * @return
     */
    private static long getPercentile(List<TaskState> tasks, int percentileRatio) {
        if (tasks.size() == 0) {
            return 0;
        }

        Collections.sort(tasks, new Comparator<TaskState>() {
            @Override
            public int compare(TaskState o1, TaskState o2) {
                return (int) (duration(o1) - duration(o2));
            }
        });
        int indexOfPercentile = Long.valueOf(Math.round(((double)percentileRatio * tasks.size() / 100) + 0.5)).intValue();
        if (indexOfPercentile > 0) {
            indexOfPercentile--;
        }

        final TaskState task = tasks.get(indexOfPercentile);
        return duration(task);
    }

    private static Report generateReportData(Map<String, Long> monthPercentileMap) {
        final Report report = new Report();
        ReportTimeScale timeScale = findTimeScale(monthPercentileMap);
        for (String month : monthPercentileMap.keySet()) {
            ReportLine line = report.addLine();
            line.addValue("month", month);
            line.addValue("timeScale", timeScale.toString());
            switch (timeScale) {
            case SECS:
                line.addValue("deploymentTime", formatToSecs(monthPercentileMap.get(month)));
                break;
            case MINS:
                line.addValue("deploymentTime", formatToMins(monthPercentileMap.get(month)));
                break;
            default:
                line.addValue("deploymentTime", formatToHours(monthPercentileMap.get(month)));
            }
        }
        return report;
    }

    private static ReportTimeScale findTimeScale(Map<String, Long> monthPercentileMap) {
        Long maxDeploymentTime = Collections.max(monthPercentileMap.values());
        if (TimeUnit.MILLISECONDS.toSeconds(maxDeploymentTime) < 600) {
            return ReportTimeScale.SECS;
        } else if (TimeUnit.MILLISECONDS.toMinutes(maxDeploymentTime) < 600) {
            return ReportTimeScale.MINS;
        }
        return ReportTimeScale.HRS;
    }
}
