package com.xebialabs.deployit.core.rest.api.reports.widgets;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

import com.xebialabs.deployit.task.TaskType;
import org.joda.time.DateTime;

import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.Report.ReportLine;
import com.xebialabs.deployit.core.rest.api.reports.TaskStateData;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.archive.ArchivedTask;
import com.xebialabs.deployit.task.archive.JcrTaskArchive;
import com.xebialabs.deployit.task.archive.JcrTaskArchive.TaskCallback;

/*********************
 * Status overview widget
 */
public class DeploymentsPercentageWidgetData extends DashboardWidgetBase {

    public DeploymentsPercentageWidgetData(final JcrTaskArchive taskArchive) {
        super(taskArchive);
    }

    @Override
    public Report getReport(final DateTime beginDate, final DateTime endDate) {
        return generateReportData(groupCompletedTasksByState(getSearchParameters().createdBetween(beginDate, endDate)));
    }

    private static Report generateReportData(TaskStateData stateData) {
        final Report report = new Report();
        if (stateData.getTotalTasks() == 0) {
            return report;
        }
        ReportLine line = report.addLine();
        DecimalFormat format = new DecimalFormat("#.##", DecimalFormatSymbols.getInstance(Locale.ENGLISH));
        line.addValue("noOfSuccessfulDeployments",
                format.format(getPercentage(stateData.getNoOfSuccessfulDeployments().get(), stateData.getTotalTasks())));
        line.addValue("noOfAbortedDeployments",
                format.format(getPercentage(stateData.getNoOfAbortedDeployments().get(), stateData.getTotalTasks())));
        line.addValue("noOfFailedDeployments",
                format.format(getPercentage(stateData.getNoOfFailedDeployments().get(), stateData.getTotalTasks())));
        line.addValue("noOfRollbacks",
                format.format(getPercentage(stateData.getNoOfRollbacks().get(), stateData.getTotalTasks())));
        return report;
    }

    /**
     * Groups only completed tasks by state. Ignores the middle states like EXECUTING or PENDING.
     *
     * @return
     */
    private TaskStateData groupCompletedTasksByState(ArchivedTaskSearchParameters archivedTaskSearchParameters) {
        final TaskStateData stateData = new TaskStateData();
        taskArchive.searchTasksWithoutLoadingSteps(archivedTaskSearchParameters, new TaskCallback() {
            public void doWithTask(ArchivedTask task) {

                if(task.getMetadata() != null &&
                        task.getMetadata().get("taskType") != null &&
                        task.getMetadata().get("taskType").equals(TaskType.ROLLBACK.toString())){
                    stateData.getNoOfRollbacks().incrementAndGet();
                } else {
                    switch (task.getState()) {
                    case DONE:
                        if (task.getFailureCount() == 0) {
                            stateData.getNoOfSuccessfulDeployments().incrementAndGet();
                        } else
                            stateData.getNoOfFailedDeployments().incrementAndGet();
                        break;
                    case CANCELLED:
                        stateData.getNoOfAbortedDeployments().incrementAndGet();
                        break;
                    }
                }
            }
        });
        return stateData;
    }


}
