package com.xebialabs.deployit.core.rest.resteasy;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.io.TemporaryFiles;
import org.apache.james.mime4j.storage.AbstractStorageProvider;
import org.apache.james.mime4j.storage.Storage;
import org.apache.james.mime4j.storage.StorageOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;

public class Mime4jStorageProvider extends AbstractStorageProvider {

    private static final String DEFAULT_PREFIX = "m4j";

    private final String prefix;
    private final String suffix;
    private final File directory;

    /**
     * Equivalent to using constructor
     * <code>TempFileStorageProvider("m4j", null, null)</code>.
     */
    public Mime4jStorageProvider() {
        this(DEFAULT_PREFIX, null, null);
    }

    /**
     * Equivalent to using constructor
     * <code>TempFileStorageProvider("m4j", null, directory)</code>.
     */
    public Mime4jStorageProvider(File directory) {
        this(DEFAULT_PREFIX, null, directory);
    }

    /**
     * Creates a new <code>TempFileStorageProvider</code> using the given
     * values.
     *
     * @param prefix
     *            prefix for generating the temporary file's name; must be at
     *            least three characters long.
     * @param suffix
     *            suffix for generating the temporary file's name; may be
     *            <code>null</code> to use the suffix <code>".tmp"</code>.
     * @param directory
     *            the directory in which the file is to be created, or
     *            <code>null</code> if the default temporary-file directory is
     *            to be used (specified by the system property
     *            <code>java.io.tmpdir</code>).
     * @throws IllegalArgumentException
     *             if the given prefix is less than three characters long or the
     *             given directory does not exist and cannot be created (if it
     *             is not <code>null</code>).
     */
    public Mime4jStorageProvider(String prefix, String suffix, File directory) {
        if (prefix == null || prefix.length() < 3)
            throw new IllegalArgumentException("invalid prefix");

        if (directory != null && !directory.isDirectory()
                && !directory.mkdirs())
            throw new IllegalArgumentException("invalid directory");

        this.prefix = prefix;
        this.suffix = suffix;
        this.directory = directory;
    }

    public StorageOutputStream createStorageOutputStream() throws IOException {
        File file = File.createTempFile(prefix, suffix, directory);
        file.deleteOnExit();

        return new TempFileStorageOutputStream(file);
    }

    private static final class TempFileStorageOutputStream extends
            StorageOutputStream {
        private File file;
        private OutputStream out;

        public TempFileStorageOutputStream(File file) throws IOException {
            this.file = file;
            this.out = new FileOutputStream(file);
        }

        @Override
        public void close() throws IOException {
            super.close();
            out.close();
        }

        @Override
        protected void write0(byte[] buffer, int offset, int length)
                throws IOException {
            out.write(buffer, offset, length);
        }

        @Override
        protected Storage toStorage0() throws IOException {
            // out has already been closed because toStorage calls close
            return new Mime4jTempFile(file);
        }
    }

    /**
     * Adaption of the standard TempFileStorage found in Mime4j's TemporaryFileStorageProvider.
     * It keeps a reference to all the streams that have been opened on the file, so that they
     * can be closed before deleting the file.
     */
    public static final class Mime4jTempFile implements Storage, TemporaryFiles.TemporaryFile {

        private File file;

        private Set<InputStream> streamsOpened = newHashSet();

        public Mime4jTempFile(File file) {
            this.file = file;
            // Register me for deletion upon thread exit.
            TemporaryFiles.register(this);
            logger.debug("Registered temporary file: {}", this);
        }

        public void delete() {
            if (file == null) return;

            logger.debug("Deleting temporary file: {}", this);

            for (InputStream inputStream : streamsOpened) {
                Closeables.closeQuietly(inputStream);
            }
            streamsOpened.clear();

            if (file != null) {
                if (file.delete()) {
                    file = null;
                }
            }
        }

        public boolean deleteWithResponse() {
            delete();
            return file == null;
        }

        public InputStream getInputStream() throws IOException {
            if (file == null)
                throw new IllegalStateException("temporary file has been deleted.");

            BufferedInputStream bufferedInputStream = new BufferedInputStream(new FileInputStream(file));
            streamsOpened.add(bufferedInputStream);
            return bufferedInputStream;
        }

        @Override
        public String toString() {
            return "Mime4jTempFile[" + (file == null ? null : file.toString()) + "]";
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(Mime4jStorageProvider.class);
}
