package com.xebialabs.deployit.core.rest.resteasy;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.ext.Provider;
import org.jboss.resteasy.annotations.interception.ServerInterceptor;
import org.jboss.resteasy.core.ResourceMethod;
import org.jboss.resteasy.core.ServerResponse;
import org.jboss.resteasy.spi.Failure;
import org.jboss.resteasy.spi.HttpRequest;
import org.jboss.resteasy.spi.interception.PostProcessInterceptor;
import org.jboss.resteasy.spi.interception.PreProcessInterceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.io.TemporaryFiles;
import com.xebialabs.deployit.jcr.JcrTemplateHolder;
import com.xebialabs.deployit.repository.NodeReaderContext;
import com.xebialabs.deployit.security.Permissions;

/**
 * Opens and closes Jcr Sessions and NodeReaderContexts. If an exception occurs, the postProcess isn't called, but the ClosingExceptionMapper cleans up after us in that case.
 */
@Provider
@Component
@ServerInterceptor
public class OpenJcrSessionDuringRequestInterceptor implements PreProcessInterceptor, PostProcessInterceptor {
    public static final String MDC_KEY_USERNAME = "username";

    @Override
    public void postProcess(ServerResponse response) {
        logger.trace("Logging out...");
        JcrTemplateHolder.get().logout();
        NodeReaderContext.get().release();
        Permissions.clearSecurityContext();
        TemporaryFiles.clear();
        WorkdirHolder.clear();
        MDC.remove(MDC_KEY_USERNAME);
    }

    @Override
    public ServerResponse preProcess(HttpRequest request, ResourceMethod method) throws Failure, WebApplicationException {
        JcrTemplateHolder.get().login();
        NodeReaderContext.get().hold();

        if (method.getResourceClass().isAnnotationPresent(Workdir.class)) {
            WorkdirHolder.initWorkdir();
        }

        Authentication authentication = Permissions.getAuthentication();
        if (authentication != null) {
            MDC.put(MDC_KEY_USERNAME, authentication.getName());
        }
        return null;
    }

    private static final Logger logger = LoggerFactory.getLogger(OpenJcrSessionDuringRequestInterceptor.class);
}
