package com.xebialabs.deployit.core.rest.xml;

import java.util.Collection;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.engine.xml.XStreamProvider;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.MethodDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static com.xebialabs.deployit.core.rest.xml.Converters.bool;
import static com.xebialabs.deployit.engine.xml.Converters.writeNode;

@XStreamProvider(tagName = "descriptor", readable = Descriptor.class)
public class DescriptorWriter implements Converter {

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        Descriptor descriptor = (Descriptor) source;
        writer.addAttribute("type", descriptor.getType().toString());
        if (descriptor.getDeployableType() != null) {
            writer.addAttribute("deployableType", descriptor.getDeployableType().toString());
        }
        if (descriptor.getContainerType() != null) {
            writer.addAttribute("containerType", descriptor.getContainerType().toString());
        }
        writer.addAttribute("virtual", bool(descriptor.isVirtual()));
        if (descriptor.getRoot().getRootNodeName() != null) {
            writer.addAttribute("root", descriptor.getRoot().getRootNodeName());
        }
        writer.startNode("description");
        writer.setValue(descriptor.getDescription());
        writer.endNode();
        writePropertyDescriptors(writer, descriptor);
        writeControlTasks(writer, descriptor);
        writeInterfaces(writer, descriptor);
        writeSuperTypes(writer, descriptor);
    }

    private static void writeSuperTypes(HierarchicalStreamWriter writer, Descriptor descriptor) {
        writer.startNode("superTypes");
        for (Type type : descriptor.getSuperClasses()) {
            writer.startNode("superType");
            writer.setValue(type.toString());
            writer.endNode();
        }
        writer.endNode();
    }

    private static void writeInterfaces(HierarchicalStreamWriter writer, Descriptor descriptor) {
        writer.startNode("interfaces");
        for (Type type : descriptor.getInterfaces()) {
            writer.startNode("interface");
            writer.setValue(type.toString());
            writer.endNode();
        }
        writer.endNode();
    }

    private static void writeControlTasks(HierarchicalStreamWriter writer, Descriptor descriptor) {
        writer.startNode("control-tasks");
        for (MethodDescriptor methodDescriptor : descriptor.getControlTasks()) {
            writer.startNode("control-task");
            writer.addAttribute("name", methodDescriptor.getName());
            writer.addAttribute("fqn", methodDescriptor.getFqn());
            writer.addAttribute("description", methodDescriptor.getDescription());
            writer.addAttribute("label", methodDescriptor.getLabel());
            if (!methodDescriptor.getAttributes().isEmpty()) {
                writer.startNode("attributes");
                for (Map.Entry<String, String> entry : methodDescriptor.getAttributes().entrySet()) {
                    writeNode(entry.getKey(), entry.getValue(), writer);
                }
                writer.endNode();
            }
            writer.endNode();
        }
        writer.endNode();
    }

    private static void writePropertyDescriptors(HierarchicalStreamWriter writer, Descriptor descriptor) {
        writer.startNode("property-descriptors");
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            wirtePropertyDescriptor(writer, pd);
        }
        writer.endNode();
    }

    private static void wirtePropertyDescriptor(HierarchicalStreamWriter writer, PropertyDescriptor pd) {
        writer.startNode("property-descriptor");
        writer.addAttribute("name", pd.getName());
        writer.addAttribute("fqn", pd.getFqn());
        writer.addAttribute("label", pd.getLabel());
        writer.addAttribute("kind", pd.getKind().name());
        writer.addAttribute("description", pd.getDescription());
        writer.addAttribute("category", pd.getCategory());
        writer.addAttribute("asContainment", bool(pd.isAsContainment()));
        writer.addAttribute("inspection", bool(pd.isInspectionProperty()));
        writer.addAttribute("required", bool(pd.isRequired()));
        writer.addAttribute("requiredInspection", bool(pd.isRequiredForInspection()));
        writer.addAttribute("password", bool(pd.isPassword()));
        writer.addAttribute("transient", bool(pd.isTransient()));
        writer.addAttribute("size", pd.getSize().name());
        if (pd.isHidden()) {
            writer.addAttribute("hidden", "true");
        }
        writeDefaultAttributeForProperty(writer, pd);

        if (pd.getReferencedType() != null) {
            writer.startNode("referencedType");
            writer.setValue(pd.getReferencedType().toString());
            writer.endNode();
        }
        if (pd.getEnumValues() != null && !pd.getEnumValues().isEmpty()) {
            writer.startNode("enumValues");
            for (String s : pd.getEnumValues()) {
                writer.startNode("string");
                writer.setValue(s);
                writer.endNode();
            }
            writer.endNode();
        }
        writer.endNode();
    }

    private static void writeDefaultAttributeForProperty(HierarchicalStreamWriter writer, PropertyDescriptor pd) {
        String defaultValue = null;
        switch (pd.getKind()) {
            case BOOLEAN:
                if (pd.getDefaultValue() != null && (Boolean)pd.getDefaultValue()) {
                    defaultValue = bool((Boolean) pd.getDefaultValue());
                }
                break;
            case ENUM:
                if (pd.getDefaultValue() != null) {
                    defaultValue = ((Enum<?>) pd.getDefaultValue()).name();
                }
                break;
            case STRING:
                if (Strings.isNotBlank((String) pd.getDefaultValue())) {
                    defaultValue = (String)pd.getDefaultValue();
                }
                break;
            case INTEGER:
                if (pd.getDefaultValue() != null && ((Integer)pd.getDefaultValue()) > 0) {
                    defaultValue = ((Integer)pd.getDefaultValue()).toString();
                }
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                defaultValue = render((Collection<?>) pd.getDefaultValue());
                break;
            case MAP_STRING_STRING:
                defaultValue = render((Map<?, ?>) pd.getDefaultValue());
                break;
        }
        if (defaultValue != null) {
            writer.addAttribute("default", defaultValue);
        }
    }

    private static String render(Collection<?> items) {
        if (items == null || items.isEmpty()) {
            return null;
        }

        return Joiner.on(", ").join(items);
    }

    private static String render(Map<?, ?> items) {
        if (items == null || items.isEmpty()) {
            return null;
        }

        return Joiner.on(", ").withKeyValueSeparator(":").join(items);
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        throw new IllegalStateException("Cannot unmarshal descriptors");
    }

    @Override
    public boolean canConvert(@SuppressWarnings("rawtypes") Class type) {
        return Descriptor.class.isAssignableFrom(type);
    }

    static Logger logger = LoggerFactory.getLogger(DescriptorWriter.class);
}
